#!/usr/bin/python

import sys, string, os, time, re, shutil, commands, glob
from types import *
from call_matlab import MatlabEngine
from yn_query import yn_query

# table of contents:
# general purpose subroutines:
#   - os.path.exists
#   - runcmd
#   - run_listblocks
#   - get_time_range
#   - get_data2load_timerange
#   - ddtime
# adcp processing subroutines
#   - zapsetup
#   - asetup
#   - aflagit_setup
#   - scanping [scanraw]
#   - loadping [loadlta, loadens, loaduh + ldcodas]
#   - setflags
#   - ubprint  [catnav]
#   - rotate
#   - xducerxy 
#   - aedit
#   - as_nav 
#   - refabs
#   - refplot
#   - smoothr, refsm
#   - putnav
#   - lst_temp
#   - lst_btrk, refabsbt, btcaluv
#   - timslip, adcpcal
#   - plottemp
#   - timegrid, as_vect; llgrid, as_cont



### =====================  general purpose subroutines ======================

# for use with runcmd (calls yn_query)
def def_qlist(question=None,  auto=None):
   ## default query. override components below if desired
   if (auto is None):
      print 'debug: must specify default answer as second argument'
      sys.exit()
   if (auto == 'q'):
      print 'quitting'
      sys.exit()
      
   if (auto == 0 or auto == 'n'):
      qa_str = 'n'
      qa_int = 0
   elif (auto == 1 or auto == 'y'):
      qa_str = 'y'
      qa_int = 1
   else:
      print 'debug: default must be integer 0 or 1, or character "y", "n", or "q"'
      sys.exit()
   
   return  (yn_query,
            [question, 'ynq'],
            {'default':qa_str, 'auto':qa_int})

def yeslist():
   return (yn_query, ['',''],  {'default':'y', 'auto':1} )   #force 'yes'   

#--- end runcmd helpers ------------------------------------------------

def file_msg(filename):
   # send a useful message to the user: 'this file exists' or 'making this file'
   if os.path.exists(filename):
      print '%s exists' % filename
   else:
      print 'making %s' % filename

#--- end file_msg -------------------------------------------------------

def get_filelist(opts):
   #make sure opts['filelist'] has a list of valid files
   # NOTE: these files should be valid from the scan or load directory, but
   # if they come in from the command line, they may be relative or
   # full paths, relative to procdir (not procdir/scan)
   
   startdir = os.getcwd()
   workdir = 'scan'
   
   
   os.chdir(workdir)
   
   if len(opts['filelist']) == 0:
      pglob = os.path.join(opts['datadir'], opts['datafile_glob'])
      opts['filelist'] = glob.glob(pglob)
      opts['filelist'].sort()

      #print '----> datadir is ', opts['datadir']
      #print '----> datafile_glob is ', opts['datafile_glob']
      #print '----> pglob is ', pglob
      #sys.exit()

      print 'found %d files using wildcard expansion from "%s"' \
            % (len(opts['filelist']), pglob)
   else:
      if opts['verbose']:
         print 'original filelist is :\n' + \
               string.join(opts['filelist'], '\n') + '\n\n'
      # fullpath file list
      ffilelist = [f for f in opts['filelist'] if f[0] == '/']
      if opts['verbose']:
         print 'fullpath filelist is :\n' + string.join(ffilelist, '\n') + '\n\n'
      # relative file list
      rfilelist = [f for f in opts['filelist'] if f[0] != '/']
      #now make the final file list suitable relative to scan/ or load/
      for ii in range(0, len(rfilelist)):
         rfilelist[ii] = os.path.join(opts['datadir'],rfilelist[ii])
      if opts['verbose']:
         print 'relpath filelist is :\n' + string.join(rfilelist, '\n') + '\n\n'
      opts['filelist'] =  ffilelist + rfilelist
      

      badfiles = [f for f in opts['filelist'] if not os.path.exists(f)]
      if len(badfiles) > 0:
         print 'in %s' % os.getcwd()
         print 'WARNING: relative to scan/'
         print 'these files from the command line do not exist:', badfiles
         os.chdir(startdir)
         sys.exit()
         
      else:
         print 'found %d files from original filelist' %len(opts['filelist'])
         #print 'using data files (as seen from scan/ or load/ directory):\n' + \
         #      string.join(opts['filelist'], '\n') + '\n\n'

   os.chdir(startdir)
            

#--- end get_filelist -------------------------------------------------------

def list2matcell(filelist):
   matstr =   '{\'' + string.join(filelist, '\',...\n\'') + '\'}'
   return matstr

#--- end matcellstr -------------------------------------------------------

def runcmd(pq=None, pycmd=None, sq=None, syscmd=None, verbose=0, workdir=None, logfile=None):
   #       python query and command; system query and command
   #
   # get cwd, chdir to working dir, do what is requested in order:
   #   ask whether to run the python command
   #   run python command (eg, make scanping.tmp
   #   as whether to run the system command
   #   run system command (eg scanping scanping.tmp)
   # chdir back to startdir
   #### NOTE: structure of pycmd is (py_fcn_to_call, (tuple), {dict})
   ####       example: a python program that only takes one positional argument:
   ####                  (asetup, (opts, ))
   ####       example: a python program that only takes named arguments:
   ####                  (function, (), dict)                   #should work


   if verbose:
      print '---------------begin verbose in runcmd--------------------'
      print 'startdir is %s' % os.getcwd()
      print 'workdir  is %s' % workdir 
      print 'pq  is:',  pq
      print 'pycmd is',  pycmd
      print 'sq  is:', sq
      print 'syscmd is',  syscmd
      print '---------------- end verbose in runcmd--------------------'

   if (pycmd and not pq) or (syscmd and not sq):
      print 'you must ask before using the command'
      print 'but you CAN specify the answer  :)'
      sys.exit()


   ## start the list of strings which will be appended to the logfile
   loglist = ['\n', time.asctime()]

   startdir = os.getcwd();
   print 'starting in %s' %startdir

   if workdir:
      os.chdir(workdir)
   ss =  'executing commands from %s' %os.getcwd()
   print ss; loglist.append(ss)

   ###
   if pq:
      qa = apply(apply, pq)
      if (qa and pycmd): #if there is a python command, run it
         ss =  'running python command "%s"' %repr(pycmd[0]).split(' ')[1]
         print ss; loglist.append(ss)
         apply(apply, pycmd)

   ###
   if sq:
      qa = apply(apply, sq)
      if (qa and syscmd): #if there is a system command, run it
         if type(syscmd) is StringType:
            ss =  'about to run %s' %syscmd
            print ss; loglist.append(ss)
            os.system(syscmd)
         elif type(syscmd) is ListType:
            for ii in range (0, len(syscmd)):
               ss =  'about to run %s' %syscmd[ii]
               print ss; loglist.append(ss)
               os.system(syscmd[ii])
         else:
            ss =  'error: syscmd must be a string or a tuple'
            print ss
            sys.exit(1)

   ###      
   if workdir:
      os.chdir(startdir)

   logappend(logfile, loglist)

#------ end runcmd ---------------------------------------------------

def run_listblocks(opts):

   dbdir = 'adcpdb'
   startdir = os.getcwd()
   #print 'about to run_listblocks from %s' %startdir
   #print 'changing directories to %s' % dbdir
   os.chdir(dbdir)

   ## (1)  run mkblkdir
   dirblk = opts['dbname'] + 'dir.blk'
   if os.path.exists(dirblk):
      os.remove(dirblk)
      print 'removing %s' % dirblk

   db_files = glob.glob('%s*.blk' % opts['dbname'])
   #print 'about to run mkblkdir on these files %s' % db_files
   print 'about to run mkblkdir'
   if len(db_files) > 0:
      ## There are still blocks left, so make a new blkdir file
      print 'database still exists; rerunning mkdirblk'
      
      cntfile = "mkblkdir.tmp";
      mkblkdir = ['DB_NAME %(dbname)s' % opts]
      mkblkdir.append('end')
      
      for dbfile in db_files:
         mkblkdir.append('BLOCK_FILE %s' % dbfile)
         
         mk = open(cntfile, 'w')
         mk.write('\n'.join(mkblkdir))
         mk.close()
         
         syscmd = '%s %s' % (os.path.join(opts['bindir'], 'mkblkdir'),
                             cntfile)
      print  'running %s' %syscmd
      os.system(syscmd)

   ## (2)  run list_blk
   outfile = '%s.lst' % opts['afilebase']
   syscmd= '%s %s %s' %  (os.path.join(opts['bindir'], 'lstblock'),
                          opts['dbname'], outfile)
   os.system(syscmd)

   # (3) cd back, update time range
   os.chdir(startdir)
   get_lstblk_tr(opts)
   
#--- end run_listblocks -------------------------------------------------
## get time range from block list  ## should be replaced by get_time_range
def get_lstblk_tr(opts):
   workdir = 'adcpdb'
   trfile = '%(afilebase)s.tr' % opts
   timefile = '%(afilebase)s.lst' % opts
   try:
      os.remove(os.path.join(workdir,trfile))
      print 'deternining time range from lstblk output'
   except:
      print 'could not update timerange file from listblocks'
      pass
   print 'writing time range from %s to %s' %(timefile, trfile)
   pycmd = [get_time_range, (opts, timefile, trfile)]
   ## list of start_end_time, dd0, dd1 available through opts['returnval']
   ## these refer to the current contents of the database
   runcmd(pq=yeslist(), pycmd=pycmd,  workdir=workdir, logfile=opts['logfile'])

##--- end get_lstblk_tr ---------------------------------------------------

def get_remaining_times(opts):

   blkfiles = glob.glob(os.path.join('adcpdb', opts['dbname'] + '*.blk'))
   #print 'listing blocks; getting current database times. found %s' % blkfiles
   print 'getting current (remaining after deletion) database times'
   if len(blkfiles) > 1:
      run_listblocks(opts)
      # get time range from lstblock
      trfile = os.path.join('adcpdb','%(afilebase)s.tr' % opts)
      timefile = os.path.join('adcpdb', '%(afilebase)s.lst' % opts)
      get_time_range(opts, timefile, trfile)
      opts['remaining_time_range'] = opts['returnval'][0]
      opts['remaining_startdd']    = opts['returnval'][1]
      opts['remaining_enddd']      = opts['returnval'][2]
   else:
      opts['remaining_time_range'] = '1970/01/01  01:00:00 to 1970/01/01  01:01:01'
      opts['remaining_startdd'] = -10000
      opts['remaining_enddd'] =   -10001


#---- end get_remaining_times --------------------------------------------

def get_time_range(opts, timefile, trfile):
   # This function reads a file from scan or lstblock and returns the ascii
   # time_range and decimal start and end days of the data
   
   pat = r"\b\d{2,4}/\d{2}/\d{2}\s+\d{2}:\d{2}:\d{2}\b"      

   if (os.path.exists(trfile) == 0):
      TF = open(timefile,'r')
      first = 0
      timelines = TF.readlines()
      TF.close()
      first = 1
      for ii in range(0, len(timelines)):
         matchobj = re.findall(pat, timelines[ii])
         if len(matchobj) >0: 
            timestamp = matchobj[0]
            if first:
               start_time = timestamp
               first = 0
            if len(matchobj) == 1:        #scanping: one date stamp per line
               end_time = matchobj[0]
            else:                         #scan from rawadcp read  or lstblk
               end_time = matchobj[1]

      start_end_times = start_time+' to '+end_time
      print 'extracting  time range'

      TR = open(trfile,'w')
      TR.write(start_end_times)
      TR.write('\n')
      TR.close()
   else:
      TR = open(trfile,'r')
      start_end_times = TR.readline()
      TR.close()
      print 'reading time range'

   #now get dday
   matchobj = re.findall(pat,start_end_times);
   start_time = matchobj[0]
   end_time = matchobj[1]

   dd0 = ddtime(start_time, str(opts['yearbase']))
   dd1 = ddtime(end_time, str(opts['yearbase']))
   #return start_end_times

   opts['returnval'] = [start_end_times, dd0, dd1]

#----- end  get_time_range -------------------------------------------------

def ddtime(date_time, int_ybase):
   # This function returns yearday when given date/time in standard format
   yearbase = str(int_ybase)
   datepat = r"\d{2,4}/\d{2}/\d{2}"
   timepat = r"\d{2}:\d{2}:\d{2}"
   ymd = re.findall(datepat, date_time)[0].split('/')
   hms = re.findall(timepat, date_time)[0].split(':')

   if (len(yearbase) == 4): yearbase_4 = yearbase 
   elif (len(yearbase) == 2):
      if atoi(yearbase) > 10: yearbase_4 = '19'+yearbase
      else: yearbase_4 = '20'+yearbase
   else:
      yb = ymd[0]
      if (len(yb) == 4): yearbase_4 = yb
      elif atoi(yb) > 10: yearbase_4 = '19'+yb
      else: yearbase_4 = '20'+yb


   sectime = time.mktime([int(ymd[0]),
                          int(ymd[1]), 
                          int(ymd[2]),
                          int(hms[0]),
                          int(hms[1]), 
                          int(hms[2]), 0, 0, 0])
   secbase = time.mktime([int(yearbase_4), 1, 1, 0, 0, 0, 0, 0, 0])
   dd = (sectime - secbase)/(24*3600)
   return dd


#----- end ddtime---------------------------------------------------------

def get_cmd_timerange(filename):
   #read the specified cmd file and get the first profile and last ascii time

   pat = r"\b\d{2,4}/\d{2}/\d{2}\s+\d{2}:\d{2}:\d{2}\b"
   fid = open(filename,'r')
   lines = fid.readlines()  # read some lines; don't need first one
   fid.close()

   start_time = ''
   end_time = ''

   for iline in range(0,50):
      if lines[iline].split(':')[0] == 'new_profile':
         matchobj = re.findall(pat, lines[iline])
         if len(matchobj) > 0:
            start_time = matchobj[0]
            break
      
   fliplines = []
   for iline in range(0,50):
      fliplines.append(lines[-(iline+1)])
   for iline in range(0,50):
      if fliplines[iline].split(':')[0] == 'new_profile':
         matchobj = re.findall(pat, fliplines[iline])
         if len(matchobj) > 0:
            end_time = matchobj[0]
            break

   return [start_time, end_time]

#----- end get_cmd_timerange --------------------------------------------------


def get_data2load_timerange(opts):

   ## run this from procdir

   ## possibilities are:
   # (1) uhdas (first time in unloaded cmd files through end of cmd files)
   # (2) ens, enx (all scanned)
   # (3) pingdata, sta, or vmdas (scanned time range, but raise error
   #                                if scan begins before codasdb ends)

   ## get the numbered block files
      
   print 'codas database currently goes from %s' % opts['remaining_time_range']
   print ' (%f to %f)\n\n' % (opts['remaining_startdd'], opts['remaining_enddd'])

   if opts['datatype'] == 'uhdas':
      blkfiles = glob.glob(os.path.join('adcpdb', opts['dbname'] + '*.blk'))
      blkstrs = []  ## 001 002 003, etc
      for blkfile in blkfiles:
         ss = blkfile[-7:-4]
         if ss != 'dir':
            blkstrs.append(ss)
      print 'found existing adcpdb blocks: '
      print blkstrs, '\n'

      cmd_blks = glob.glob(os.path.join('load', 'ens_blk*.cmd'))
      cmd_strs = []  ## 001 002 003, etc
      for cmdfile in cmd_blks:
         cmd_strs.append(cmdfile[-7:-4])

      loaded_str = []
      unloaded_str = []
      for ss in cmd_strs:
         if ss in blkstrs:
            loaded_str.append(ss)
         else:
            unloaded_str.append(ss)

      print 'found existing ens_blk files: '
      print cmd_strs, '\n'

      print 'checked: loaded blkstrs: ', loaded_str
      print 'checked: unloaded blkstrs: ', unloaded_str

      if  opts['scn_enddd'] <= opts['remaining_startdd']:
         # no data to load
         opts['thisload_time_range'] = '';
         opts['thisload_startdd'] = None
         opts['thisload_enddd'] =   None
      else:
         if len(unloaded_str) > 0:
            ## start time of unloaded data from first unloaded file
            unloaded_file = os.path.join('load', 'ens_blk%s.cmd' % (unloaded_str[0]))
            print 'first unloaded file is %s' % (unloaded_file)         
            [starttime, tr1] = get_cmd_timerange(unloaded_file)
            opts['thisload_startdd'] =  ddtime(starttime,  str(opts['yearbase']))
            opts['thisload_enddd']      = opts['scn_enddd']
            endtime = opts['scn_time_range'].split('to')[1]
            opts['thisload_time_range'] = '%s to %s' % (starttime, endtime)
         else:
            # no data to load
            opts['thisload_time_range'] = '';
            opts['thisload_startdd'] = None
            opts['thisload_enddd'] =   None

            
      if  opts['thisload_startdd'] != None:
         print 'data goes from  %s' % opts['thisload_time_range']
         print ' (%f to %f)' % (opts['thisload_startdd'], opts['thisload_enddd'])
      else:
         print 'no additional data in cmd files to load'
         
   else:
      opts['thisload_time_range'] = opts['scn_time_range']
      opts['thisload_startdd']    = opts['scn_startdd']
      opts['thisload_enddd']      = opts['scn_enddd']


#---- end get_data2load_timerange -------------------------


def flankline(string):
   # for ease of reading gobs of output to stdout
   return '\n------------ ' + string + ' -------------\n'

   
#----- end flankline ----------------------------------------------------

def logappend(logfile=None, loglist=None):
   # append a list to the logfile
   if logfile:
      logw = open(logfile, 'a')
      for ii in range(0, len(loglist)):
         logw.write(loglist[ii] + '\n')
      logw.close()

#----- end logappend ----------------------------------------------------

##===============    adcp processing routines =====================


def zapsetup(opts):

   shutil.copyfile('setup.m', 'setup.m.demo');

   setup = open('setup.m__','r')
   setup_list = setup.readlines()
   setup.close()

   for ii in range(0, len(setup_list)):
      setup_list[ii] = setup_list[ii].replace('__dbname__',     opts['dbname'])   
      setup_list[ii] = setup_list[ii].replace('__yearbase__',   str(opts['yearbase']))
      setup_list[ii] = setup_list[ii].replace('__pgmin__',      str(opts['pgmin']))
      setup_list[ii] = setup_list[ii].replace('__numbins__',    str(opts['numbins']))
      setup_list[ii] = setup_list[ii].replace('__rl_startbin__',str(opts['rl_startbin']))
      setup_list[ii] = setup_list[ii].replace('__rl_endbin__',  str(opts['rl_endbin']))

   setout = open('setup.m', 'w')
   for ii in range(0, len(setup_list)):
      setout.write(setup_list[ii])
   setout.close()

#--- end zapsetup.m -------------------------------------------------------


def asetup(opts):

   asetup = open('asetup.m__','r')
   asetup_list = asetup.readlines()
   asetup.close()

   # find and substitute for all instances indicated by __XXXX__
   for ii in range(0, len(asetup_list)):
      asetup_list[ii] = asetup_list[ii].replace('__navfile__',  '../nav/'+opts['fixfile'])
      asetup_list[ii] = asetup_list[ii].replace('__dbname__',     opts['dbname'])   
      asetup_list[ii] = asetup_list[ii].replace('__comment__',    opts['cruisename']) 
      asetup_list[ii] = asetup_list[ii].replace('__yearbase__',   str(opts['yearbase']))
      asetup_list[ii] = asetup_list[ii].replace('__numbins__',    str(opts['numbins']))
      asetup_list[ii] = asetup_list[ii].replace('__rl_startbin__',str(opts['rl_startbin']))
      asetup_list[ii] = asetup_list[ii].replace('__rl_endbin__',  str(opts['rl_endbin']))
      
   asetout = open('asetup.m','w')
   for ii in range(0, len(asetup_list)):
      asetout.write(asetup_list[ii])
   asetout.close()


#--- end asetup -------------------------------------------------------



def aflagit_setup(opts):

   afsetup = open('aflagit_setup.m__','r')
   afsetup_list = afsetup.readlines()
   afsetup.close()

   # find and substitute for all instances indicated by __XXXX__
   for ii in range(0, len(afsetup_list)):
      afsetup_list[ii] = afsetup_list[ii].replace('__pgmin__',      str(opts['pgmin']))
      afsetup_list[ii] = afsetup_list[ii].replace('__yearbase__',   str(opts['yearbase']))
      afsetup_list[ii] = afsetup_list[ii].replace('__rl_startbin__',str(opts['rl_startbin']))
      afsetup_list[ii] = afsetup_list[ii].replace('__rl_endbin__',  str(opts['rl_endbin']))
      
   afsetout = open('aflagit_setup.m', 'w')
   for ii in range(0, len(afsetup_list)):
      afsetout.write(afsetup_list[ii])
   afsetout.close()


#--- end aflagit_setup -------------------------------------------------------


def update_gbin(opts):

   ## use this list to determine how many gbin files get remade:

   opts['cellstr'] = list2matcell(opts['scanfilelist'])

   matstr = '''

   %(displaystr)s

   gbinlist = %(cellstr)s;
   cfg =get_uhdascfg('cfgbase', '%(cruisename)s', 'cfgpath', 'config');

   %% update gbin files
   update_gbin(cfg,'whichones', gbinlist , 'logfpath', '.')

''' % opts

   mat = open('updategbin_tmp.m', 'w')
   mat.write(matstr)
   mat.close()

   try:
      (runstr, output) = opts['M'].matlab('updategbin_tmp', verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
      print 'CANNOT UPDATE GBIN FILES'
      sys.exit()

   print '-----------------------------------------------'
   print '\n\n  updating gbin files from scanfilelist:'
   print string.join(opts['scanfilelist'], '\n') + '\n\n'   
   print '-----------------------------------------------'

   del opts['cellstr']


#--- end update_gbin --------------------------------------------------------


def copy_ensfile(opts):

   fin = open(opts['enddday_file'],'r')
   lines = fin.readlines()
   fin.close()
   fout = open('ensemble.enddday','w')
   for ii in range(0, len(lines)):
      fout.write(lines[ii])
   fout.close()
   
#--- end copy_ensfile -------------------------------------------------------


def scanping(opts, cntfile):
   # This function scans the pingdata files
   
   scanping_cnt = '''
   OUTPUT_FILE:         %(afilebase)s.scn
   SHORT_FORM:          yes
   UB_OUTPUT_FILE:      none
   USER_BUFFER_TYPE:    %(ub_type)s
   UB_DEFINITION:       ub_%(ub_type)s.def
   PINGDATA_FILES:
''' % opts
   
   for scanfilename in opts['scanfilelist']:
      scanping_cnt = scanping_cnt + scanfilename + '\n'

   print scanping_cnt
   scn = open(cntfile, 'w')
   scn.write(scanping_cnt)
   scn.close()

#--- end of scanping -------------------------------------------------------

def scanraw(opts):
   # this function scans uhdas and vmdas files

   opts['cellstr'] = list2matcell(opts['scanfilelist'])

   # figure out the configuration information

   # now fill in time string
   if opts['datatype'] in ('lta','sta', 'ens', 'enx'):
      opts['timestring'] = "'timetype', 'vmdas',..."
   else:
      opts['timestring'] = '''    'timegbindir', ...
      fullfile(gbindirbase, instname, 'time'),...
     'timetype', 'uhdas',... ''' % opts   


   # now fill in the matlab string
   matstr = '''

   %(displaystr)s
   

   %% set up configuration:
   %(matcfgstr)s

   filelist = %(cellstr)s;
   scan(instclass, ...
   'outfile', '%(afilebase)s.scn',...
   'filelist', filelist,...
   %(timestring)s
   'yearbase', %(yearbase)d);
''' % opts

   mat = open('scan_tmp.m', 'w')
   mat.write(matstr)
   mat.close()

   try:
      (runstr, output) = opts['M'].matlab('scan_tmp', verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
         
   del opts['cellstr']
   del opts['timestring']

#--- end scanraw ----------------------------------------------------

def overlapping(opts):
   # return 0 if "no overlap"          (i.e. loading now is OK)
   # return 1 if "there is an overlap" (i.e. do not load now)
   ## this function just does the deletion and returns 0 or 1
   ## it does not calculate the final data ranges
   ## see get_data2load_timerange()

   # run from procdir

   
   timepat = r"\b\d{2,4}/\d{2}/\d{2}\s+\d{2}:\d{2}:\d{2}"
   filepat = r"[a-zA-Z0-9_]+\d{3}.blk"

   # run this from startdir
   loglist = ['\n', time.asctime()]

   ##### (2b): list blocks to find whole timerange
   ss = 'checking database time range by listblk'
   print ss; loglist.append(ss)
   run_listblocks(opts)

   ## (2c) get filenames and timeranges from lst file
   outfile = os.path.join('adcpdb', '%s.lst' % opts['afilebase'])
   TM = open(outfile,'r')
   lines = TM.readlines()
   TM.close()
   filelist = []
   starttimelist = []
   endtimelist = []
   for ii in range(2, len(lines)):
      filematch = re.findall(filepat, lines[ii])
      timematch = re.findall(timepat, lines[ii]) #get starttime and endtime

      filelist.append(filematch[0])
      starttimelist.append(timematch[0]) # start times of data files
      endtimelist.append(timematch[1]) # end times of data files
      ss = 'found line %s' %lines[ii][:-1]
      print ss; loglist.append(ss)

   codas_endtime = timematch[1]
   codas_enddd = ddtime(codas_endtime, opts['yearbase'])

   # new data time range
   scn_starttime =  re.findall(timepat, opts['scn_time_range'])[0]
   scn_startdd = ddtime(scn_starttime, opts['yearbase'])
   opts['scn_startdd'] = scn_startdd

   ss = '\nscan data start at %10.7f '  %scn_startdd
   print ss; loglist.append(ss)
   ss = 'scan time range: %s' %opts['scn_time_range']
   print ss; loglist.append(ss)


   ss = 'codasdb ends at   %10.7f (%s)\n' %(codas_enddd, codas_endtime)
   print ss; loglist.append(ss)
   
   needs_reload = (scn_startdd < codas_enddd)


   ## change directories if necessary to delete block files
   dbdir = 'adcpdb'
   
   if needs_reload and not opts['allow_reload']:    ## simply notify 
      return 1 #not deleted, so "overlapping"
   if needs_reload and opts['allow_reload']:    ## if manual, ask; if auto force it
      if (opts['auto'] == 1):
         qa_default = 'y'
      else:
         qa_default = 'n'

      
      qa = yn_query('delete trailing blk files and load new data files?' , 'ynq',
                    default=qa_default, auto=opts['auto'])

      if qa:
         startdir = os.getcwd()
         os.chdir(dbdir)
         print '... changed dirs,  in ', os.getcwd()
         ## now we have to delete blk files until codasdb_enddd < scn_startdd


         if opts['lastfiles'] != 0: # ping, lta, sta

            rm_dirblk = 0
            ss = 'length of filelist is %d\n'% len(filelist)
            print ss; loglist.append(ss)
      
            ## this is for file-driven pre-averaged data:
            for ii in range(len(filelist)-1, -1, -1):
               ss= 'ii=%d: file <%s> ends at %10.7f, new data starts at %10.7f'\
                     %(ii, filelist[ii], ddtime(endtimelist[ii], opts['yearbase']),
                       scn_startdd)
               print ss; loglist.append(ss)
               
               if ddtime(endtimelist[ii],opts['yearbase']) >= scn_startdd: 
                  ss = '...  deleting file %s' %filelist[ii]
                  print ss; loglist.append(ss)
               
                  os.remove(filelist[ii])
                  rm_dirblk = 1
                  
                  rotlog = os.path.join(startdir, 'cal', 'rotate', 'rotate.log')
                  LOG = open(rotlog,'a')
                  ss = 'deleting file %s from database (time range %s to %s\n' \
                            % (filelist[ii], starttimelist[ii], endtimelist[ii])
                  LOG.write(ss)
                  LOG.close()
               
               if rm_dirblk:
                  dirblk = opts['dbname'] + 'dir.blk'
                  ss = 'removing dir blk file %s' % dirblk
                  print ss; loglist.append(ss)
                  os.remove(dirblk)
                  ss = '%d blk files remaining for this database' \
                       %len(glob.glob(opts['dbname'] + '*.blk'))
                  print ss; loglist.append(ss)
               
               else:
                  ss = '...  keeping file %s' %filelist[ii]
                  print ss; loglist.append(ss)
                  break

            chdir(startdir)
            return 0
         
         elif  opts['delblks'] != []: # uhdas
            dirblk = opts['dbname'] + 'dir.blk'
            if os.path.exists(dirblk):
               os.remove(dirblk)
            rm_dirblk = 1
            blklist = glob.glob(opts['dbname'] + '*.blk')
            #print ' checking block files to remove.  avilable blockfiles:'
            #print blklist

            delfiles = []
            if opts['delblks'] > 0:
               for ifile in range(0, opts['delblks']):
                  delfiles.append(blklist[len(blklist) - (ifile+1)])
            print ' deleting these files %s' % delfiles
            for delfile in delfiles:
               os.remove(delfile)

            os.chdir(startdir)
            return 0
            
         else:
            print 'not attempting to delete anything'
            os.chdir(startdir)
            return 1
            
            
         if  (len(glob.glob(opts['dbname'] + '*.blk')) == 0):
            os.chdir(startdir)
            return 0      #deleted the whole thing, so start by loading

      else:
         #write to logfile and return
         logappend(opts['logfile'], loglist)
         return  1  # not deleted, so "loaded"


#--- end loaded -------------------------------------------------------
def loadping(opts, cntfile):
   # This function loads pingdata files
   
   loadping_cnt = '''
     DATABASE_NAME:            ../adcpdb/%(dbname)s
     DEFINITION_FILE:          ../adcpdb/adcp%(ub_type)s.def
     OUTPUT_FILE:              ../adcpdb/%(afilebase)s.lod
     MAX_BLOCK_PROFILES:       400
     NEW_BLOCK_AT_FILE?        yes
     NEW_BLOCK_AT_HEADER?      no
     NEW_BLOCK_TIME_GAP(min):  32767
 /*  to skip headers, insert:  skip_header_range:  n1 to n2 */
     PINGDATA_FILES:
''' % opts
   for pingfilename in opts['scanfilelist']:
      loadping_cnt = loadping_cnt + pingfilename + ' end \n'


   opts['thisload_time_range'] =   opts['scn_time_range']
   opts['thisload_startdd'] =      opts['scn_startdd']
   opts['thisload_enddd'] =        opts['scn_enddd']


   print loadping_cnt
   ld = open(cntfile, 'w')
   ld.write(loadping_cnt)
   ld.close()

#--- end loadping -------------------------------------------------------

def loadlta(opts):

   opts['cellstr'] = list2matcell(opts['scanfilelist'])


   matstr = '''

   %(displaystr)s
  

   %(matcfgstr)s
   
   files = %(cellstr)s;

   for ifile = 1:length(files)
      file = files{ifile}
      load_avg(instclass, file);
end
''' % opts
   
   mat = open('loadlta_tmp.m', 'w')
   mat.write(matstr)
   mat.close()

   try:
      (runstr, output) = opts['M'].matlab('loadlta_tmp' , verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            

   del opts['cellstr']

#--- end loadlta -------------------------------------------------------

def loadens(opts):


   opts['cellstr'] = list2matcell(opts['scanfilelist'])
   if opts['datatype'] == 'ens':
      opts['align'] = opts['ens_halign']
   else:
      opts['align'] = float(0.0)

## set up for processing
   matstr1 = '''

   %(matcfgstr)s

   %% to specify the timeranges used for averaging, load a file with
   %% ensemble times extracted from the parallel ens processing, eg:
   %% params.end_dday = read_asc('../../a0105/edit/a0105.tem', 4,1);
   if exist('ensemble.enddday','file')
      tmp=load('ensemble.enddday');
      params.end_dday = tmp(:,1);
   end


   %% if using ENS data, 
   %% You must specify the angle by which to rotate the data collected in
   %% beam coordinates into earth coordinates.  There is a parameter used
   %% during data collection:
   %%         h_align = offset_for_heading_used_during_aquisition - ...
   %%                          codas_correction_value
   %%  a decent guess for the offset_for_heading_used_during_aquisition is
   %%  the heading alignment in the concurrent LTA file.


   params.interval = %(ens_len)d;    
   params.ilayer = %(rl_startbin)d:%(rl_endbin)d;
   params.yearbase = %(yearbase)d;
   params.h_align = %(align)f;   
   params.ducer_depth = %(ducer_depth)s;
   params.velscale     = 1.0;



   params.pingedit = 'all';
   params.editcfg = get_pedefaults(instclass, pingtype);

   %% can override single ping editing either manually ...
   params.editcfg.rl_range = params.ilayer([1 end]);
   %% ... or in an m-file
   if exist('override_editcfg.m','file')
      override_editcf
   end
   
   files = %(cellstr)s;
   r = [];

   for ifile = 1:length(files)
   file = files{ifile};
   r = load_raw(instclass, file, params, ...
   'finish', 0, ...
   'pingtype', pingtype, ...
   'matfile', 1, ...
   'leftover', r);
end

''' % opts

   mat = open('loadens_tmp.m', 'w')
   mat.write(matstr)
   mat.close()
   
   try:
      (runstr, output) = opts['M'].matlab('loadens_tmp' , verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            

   del opts['cellstr']
   del opts['align']

#--- end loadens -------------------------------------------------------


def loaduh(opts):

   
   opts['cellstr'] = list2matcell(opts['filelist'])
   if opts['max_BLKprofiles'] != 399:
      opts['BLKstr'] = "'max_BLKprofiles', %d," % (opts['max_BLKprofiles'])
   else:
      opts['BLKstr'] = ' '


## set up for processing
   matstr = '''

   %(displaystr)s
   

   %(matcfgstr)s

   params = get_params(allparams, instname, pingtype);
   %% to specify the timeranges used for averaging, load a file with
   %% ensemble times extracted from the parallel ens processing, eg:
   %% params.end_dday = read_asc('../../a0105/edit/a0105.tem', 4,1);
   if exist('ensemble.enddday','file')
      tmp=load('ensemble.enddday');
      params.end_dday = tmp(:,1);
   end
  

   params.showverbose = 0;
   params.pingedit = 'all';
   params.yearbase = yearbase;

   %% can override single ping editing either manually ...
   params.editcfg.rl_range = params.ilayer([1 end]);
   %% ... or in an m-file
   if exist('override_editcfg.m','file')
      override_editcf
   end
   

   files = %(cellstr)s;  %% all files
   %% set these now (they may be reset if using num2redo 

   %% this should start with the most recent profile (from ens_blk*.log)
   %% and start there. no need to fiddle with filelists.  
   %% It is automatically in incremental mode.
   
   write_ensblk(instclass, files, params,...
       'gbindir', fullfile(gbindirbase, instname),...
       'pingtype', pingtype,  %(BLKstr)s ...
       'verbose', 0);

''' % opts


   mat = open('load_uhblk_tmp.m', 'w')
   mat.write(matstr)
   mat.close()
   
   try:
      (runstr, output) = opts['M'].matlab('load_uhblk_tmp' , verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))

   logappend(opts['logfile'], ['AVERAGING DATA... log file is load/write_ensblk.log'])

   del opts['cellstr']
   del opts['BLKstr']

#--- end loaduh -------------------------------------------------------

def ldcodas(opts, cntfile):

   ldcodas_cnt = '''
   DATABASE_NAME:   ../adcpdb/%(dbname)s
   DEFINITION_FILE: %(data_def)s
   LOG_FILE:  load.log
   YEAR_BASE: %(yearbase)d
   END

''' % opts

   print 'in %s' % os.getcwd()

   if opts['datatype'] == 'uhdas':
      blkfiles = glob.glob(os.path.join('..','adcpdb', opts['dbname'] + '*.blk'))
      blkstrs = []  ## 001 002 003, etc
      for blkfile in blkfiles:
         ss = blkfile[-7:-4]
         if ss != 'dir':
            blkstrs.append(ss)
      print 'found block strings: %s' % blkstrs

      cmd_blks = glob.glob('ens_blk*.cmd')
      cmd_strs = []  ## 001 002 003, etc
      for cmdfile in cmd_blks:
         cmd_strs.append(cmdfile[-7:-4])
      print 'found cmd blk strings: %s' % cmd_strs

      ok_cmdlist = []
      for ss in cmd_strs:
         if ss not in blkstrs:
            ok_cmdlist.append('ens_blk%s.cmd' % ss)
   else:
      ok_cmdlist = glob.glob('ens_blk*.cmd')

   print 'about to load from these cmd files %s' % ok_cmdlist

   [starttime, tr1] = get_cmd_timerange(ok_cmdlist[0])
   print 'starttime is %s' % starttime
   print 'RESETTING thisload_time_range'
   endtime = opts['scn_time_range'].split('to')[1]
   
   opts['thisload_startdd'] = ddtime(starttime, opts['yearbase'])
   opts['thisload_time_range'] = '%s to %s' % (starttime, endtime)   

   print 'time range of data to be loaded: %s' % (opts['thisload_time_range'])
   print 'data startdd %f' % opts['thisload_startdd']
   print 'data enddd %f' % opts['thisload_enddd']


   # ok_cmdlist was already trimmed
   atlist = ['@'+ ok_cmdlist[ii] for ii in range(0,len(ok_cmdlist))]
   ldcodas_cnt = ldcodas_cnt + string.join(atlist, '\n') + '\nend\n'

   cntfile = 'ldcodas.tmp'
   print ldcodas_cnt
   ld = open(cntfile, 'w')
   ld.write(ldcodas_cnt)
   ld.close()

#--- end ldcodas -------------------------------------------------------

def setflags(opts, cntfile):
   ## make and run setflags
   
   setflags_cnt = '''
   dbname:     ../adcpdb/%(dbname)s
   pg_min:      %(pgmin)s
   set_range_bit
   time_ranges:
        all
        
''' % opts

   print setflags_cnt
   sf = open(cntfile, 'w')
   sf.write(setflags_cnt)
   sf.close()


#--- end setflags --------------------------------------------------------

def lst_conf(opts, cntfile):
   ## make and run setflags
   
   lst_conf_cnt = '''
   dbname:     %(dbname)s
   output:     %(dbname)s.cnh
   time_ranges:
        all
        
''' % opts

   print lst_conf_cnt
   cf = open(cntfile, 'w')
   cf.write(lst_conf_cnt)
   cf.close()

#--- end lst_conf -------------------------------------------------------

def ubprint(opts, cntfile):
   # This function reads extracts the ags data from user-buffer in codasdb
   ub_cnt = '''
     dbname:          ../adcpdb/%(dbname)s
     output:          %(afilebase)s
     step_size=       1
     year_base=       %(yearbase)s
     variables:
        avg_GPS_summary
        avg_GPS2_summary        
        attitude_mat
        position
        end
     time_ranges:
        %(loaded_time_range)s       /* the whole time range */
        
''' % opts

   print ub_cnt
   ub = open(cntfile,'w')
   ub.write(ub_cnt)
   ub.close()

    
#--- end ubprint -------------------------------------------------------

def catnav(opts):
   #cat gps navigation from ../load/ generated by load_XXX step
   navfilelist = glob.glob('../load/*.gps2');
   fixfile = open(opts['fixfile'], 'w')
   for ii in range (0,len(navfilelist)):
      infile = open(navfilelist[ii], 'r')
      lines = infile.readlines()
      fixfile.write(''.join(lines))
      infile.close()
   fixfile.close()


#--- end catnav -------------------------------------------------------

def ashrot(opts):
   #  This routine produces a *.ang file from the matfile produced by ubprint

   matstr = '''

   %(displaystr)s
   
   fnstring = '%(afilebase)s'
   yearbase = %(yearbase)s
   cruise = '%(cruisename)s'
   ash_mat_path = '../../nav/'  %%keep trailing slash
   dh_offset = 0.;
   ashrot;

''' % opts
   
   mat = open('ashrot_tmp.m', 'w')
   mat.write(matstr)
   mat.close()

   try:
      (runstr, output) = opts['M'].matlab('ashrot_tmp', verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            

#--- end ashrot -------------------------------------------------------

def print_attlog(opts):
   #write the file with the info ------
   #changed start_dd to match decimal day (since filenames are dday+1)
   
   matstr = '''


   %(displaystr)s
   
   %%written by quick_adcp.py
   FID_LOG = []; %%needed to fake out matlab
   t0_array = %(scn_startdd)f;
   attfile = '%(afilebase)s';
   year_string = %(yearbase)d;
   start_dd = %(scn_startdd)f;
   pages = 2;
   print_flag = 0;
   dt = %(scn_enddd)f - %(scn_startdd)f;
   cruise_title = '%(cruisename)s';

   %%load the att data and set up all the variables
   if exist('load_att')
     load_att;

     for t0 = t0_array
       t1 = t0 + dt;
       time_mask = (att(i_day,:) <= t1 & att(i_day,:) >= t0);

       jall = find(time_mask);
       jgood = find(time_mask & good_mask);
       njall = length(jall);
       njgood = length(jgood);
       
       fprintf(FID_LOG,'  t0: %%5.1f;  %%d points, ',      t0, njall);
       fprintf(FID_LOG,'%%d accepted; %%5.2f percent rejected\\n', ...
       njgood, 100 * (njall - njgood)/njall);
       end
   else
      fprintf(2,'load_att not on path\\n')
   end

''' % opts


   mat = open('attlog_tmp.m', 'w')
   mat.write(matstr)
   mat.close()


   try:
      os.remove('attplot.log')      #remove the file first to avoid confusion
   except:
      pass
   # run attplot from matlab ----
   
   try:
      (runstr, output) = opts['M'].matlab('attlog_tmp', verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
         


#--- end print_attlog -------------------------------------------------------

def rotated(opts):
   # This return states whether a database has been rotated
   #  (subsequently used to determine whether further rotation should occur)
   rotated_endtime = '1970/01/01  01:01:01'
   if (os.path.exists('rotate.log') == 0):
      print 'No existing file: rotate.log'
      opts['returnval'] = 0
      return 0

   timepat = r"\b\d{2,4}/\d{2}/\d{2}\s+\d{2}:\d{2}:\d{2}"

   log = open('rotate.log','r')
   log_list = log.readlines()
   log.close()
   N = len(log_list)

   if N==0:
      print 'no valid times in rotate.log'
      opts['returnval'] = 0
      return 0

   for ii in range(0,N):
      if (string.find(log_list[ii],'Processing range') > 0):
         jj = string.find(log_list[ii],' to ')
         line = log_list[ii]
         rotated_endtime = line[jj+4:-1]

   # lastrotated_dday is end of the rotated data
   # it is either the last time in rotate.log
   # or it is the end of the originally loaded database (if that is earlier)
   lastrotated_dday = ddtime(rotated_endtime, opts['yearbase'])
   print 'last rotated time from log file: %s' % rotated_endtime
   TRdb  =  re.findall(timepat, opts['loaded_time_range'])   
   print 'time range of currently loaded data: %s to %s' %(TRdb[0], TRdb[1])
   print '(%f to %f)' %(opts['loaded_startdd'], opts['loaded_enddd'])
   # override lastrotateddday with minimum of dates from (rotate.log and db)
   tmp_dd =  ddtime(TRdb[1], opts['yearbase']);
   if (tmp_dd < lastrotated_dday):
      lastrotated_dday = tmp_dd;
      rotated_endtime = TRdb[1]
      print 'overriding last rotated time with last time in database, %s' \
            %opts['loaded_time_range']


   if opts['allow_reload']:
      ## we have reloaded data; just go ahead and allow rotate
      opts['returnval'] = 0
   else:
      # firstunrotated_dday is the start of the new data
      unrotated_starttime =  re.findall(timepat, opts['scn_time_range'])[0]
      firstunrotated_dday = ddtime(unrotated_starttime, opts['yearbase'])
      
      # return true if new block starts before end of rotated portion.
      if (firstunrotated_dday <= lastrotated_dday):
         print 'End of previously rotated range: '+rotated_endtime
         print 'Start of present range:          '+unrotated_starttime
         opts['returnval'] = 1
      else:
         opts['returnval'] = 0
         


#--- end rotated -------------------------------------------------------

def rotate(opts, cntfile):
   # The routine rotaes the velocities in the database using both the *.ang
   # file and the rotate_amplitude and rotate_angle

   if opts['no_head_corr']:
      str = '/*  time_angle_file:  (NOT USED) */'
   else:
      str = 'time_angle_file:    %(time_angle_file)s' %opts

   ## override if file does not exist
   if not os.path.exists(opts['time_angle_file']):
      str = '/* time_angle_file: %s (NOT FOUND) */' % opts['time_angle_file']
   

   opts['tempstr'] = str
   
   rotate_cnt = '''
      DB_NAME:       ../../adcpdb/%(dbname)s
      LOG_FILE:      rotate.log
      TIME_RANGE:    %(thisload_time_range)s     /* only from scanned files */

      OPTION_LIST:
         water_and_bottom_track:
           year_base=          %(yearbase)s
           %(tempstr)s
           amplitude=          %(rotate_amplitude)s
           angle_0=	          %(rotate_angle)s
           end
         end

''' % opts

   print rotate_cnt
   rot = open(cntfile,'w')
   rot.write(rotate_cnt)
   rot.close()

   del opts['tempstr']

#--- end rotate ---------------------------------------------------------
   
def xducerxy(opts):

   xducerxy_file = 'call_xducerxy.m'

   # ADD MATLAB COMMENTS IN PAIRS OF %%
   call_xducerstr = '''
      %%%% call_xducerxy being generated by quick_adcp.py 
      %%  call the engine, xducerxy_core
      xducerxy_core('yearbase',%(yearbase)d,'dy',%(xducer_dy)d,...
             'dx',%(xducer_dx)d, 'enl', %(ens_len)d, ...
             'fixfile', '%(fixfile)s');
      ''' % opts

   print 'about to run xducerxy'
   print call_xducerstr

   xydf = open(xducerxy_file, 'w')
   xydf.write(call_xducerstr)
   xydf.close()

   try:
      (runstr, output) = opts['M'].matlab('call_xducerxy', verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            
  

# --- end xducerxy -------------------------------------------------------


def as_nav(opts, cntfile):
   # This routine runs adcpsect to get navgation and heading data
   asnav_cnt = '''
      dbname:             ../adcpdb/%(dbname)s
      output:             %(afilebase)s
      step_size:          1
      ndepth:             %(rl_endbin)s
      time_ranges:        separate
      year_base=          %(yearbase)s

      option_list:
        /*pg_min=           %(pgmin)s */
        pg_min=           80
      navigation:
        reference_bins  %(rl_startbin)s  to   %(rl_endbin)s
        end
      statistics:       mean
        units=          0.01       /* cm/s instead of default m/s */
        end
      flag_mask:        ALL_BITS
        end
      end
     %(loaded_time_range)s                 /* the whole time range */

''' % opts

   print asnav_cnt
   nav = open(cntfile, 'w')
   nav.write(asnav_cnt)
   nav.close()


#--- end as_nav -------------------------------------------------------


def refabs(opts, cntfile):
   # Routine runs refabs to get the absolute reference layer velocities
   ref_cnt = '''
      fix_file_type:    simple
      reference_file:   %(afilebase)s.nav
      fix_file:         %(fixfile)s
      output:           %(afilebase)s.ref
      year_base=        %(yearbase)s
      ensemble_length=  %(ens_len)s
      gap_tolerance=    10

''' % opts
   
   print ref_cnt
   ref = open(cntfile,'w')
   ref.write(ref_cnt)
   ref.close()


    
#--- end refabs -------------------------------------------------------

def refplot(opts):

   refp = open('callrefp.m__','r')
   refp_list = refp.readlines()
   refp.close()

   # find and substitute for all instances indicated by __XXXX__
   for ii in range(0, len(refp_list)):
      refp_list[ii] = refp_list[ii].replace('__bin_dir__',opts['bindir'])
      refp_list[ii] = refp_list[ii].replace('__bin_file__',opts['afilebase']+'.bin')
      refp_list[ii] = refp_list[ii].replace('__ref_file__',opts['afilebase']+'.ref')
      refp_list[ii] = refp_list[ii].replace('__outfile__', opts['afilebase'])
      refp_list[ii] = refp_list[ii].replace('__year_base__',str(opts['yearbase']))
      refp_list[ii] = refp_list[ii].replace('__title__',opts['cruisename'])

   refpw = open('callrtmp.m','w')
   for ii in range(0, len(refp_list)):
      refpw.write(refp_list[ii]) 
   refpw.close()

   print 'Created callrtmp.m'

   matstr = '''


   %(displaystr)s
   
      for dd = floor(%(loaded_startdd)s):2:ceil(%(loaded_enddd)s);
         callrtmp(dd);
      end
      
''' % opts


   mat = open('callrtmp_tmp.m', 'w')
   mat.write(matstr)
   mat.close()

   try:
      (runstr, output) = opts['M'].matlab('callrtmp_tmp',  verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            


#--- end refplot -------------------------------------------------------


def smoothr(opts, cntfile):
   # Routine to smoorth the reference layer velocities
   smoothr_cnt = '''
      reference_file:       %(afilebase)s.nav
      refabs_output:        %(afilebase)s.ref
      output:               %(afilebase)s.sm
      filter_hwidth=        %(filter_hwidth)s
      min_filter_fraction=  0.5
      max_gap_ratio=        0.05
      max_gap_distance=     50
      max_gap_time=         30            /* 0.5  minutes */
      ensemble_time=        %(ens_len)s      
      max_speed=            5.0
      min_speed=            1.0
      iterations=           5
      fix_to_dr_limit=      0.00050

''' % opts


   print smoothr_cnt
   smth = open(cntfile, 'w')
   smth.write(smoothr_cnt)
   smth.close()

    
#--- end smoothr --------------------------------------------------------

def refsm(opts):

   refsm_file = 'refsm.m'


   # ADD MATLAB COMMENTS IN PAIRS OF %%
   refsmstr = '''
      %%%% refsm being generated by quick_adcp.py 
      yearbase = %(yearbase)d;
      dbname = '%(dbname)s';
      fixfile = '%(fixfile)s';
      %%%%  end of section to be edited by user %%
      %% now call the engine, refsm_core
      refsm_core(dbname, fixfile, yearbase, ...
         'ref_top', %(rl_startbin)d, 'ref_bot', %(rl_endbin)d);
         
''' % opts

   reff = open(refsm_file, 'w')
   reff.write(refsmstr)
   reff.close()


   try:
      (runstr, output) = opts['M'].matlab('refsm', verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            
  

# --- end refsm -------------------------------------------------------


def putnav(opts, cntfile):
   # Update the data base with the results of the reference layer smoothing

   putnav_cnt = '''
      dbname:         ../adcpdb/%(dbname)s
      position_file:  %(smfile)s
      year_base=      %(yearbase)s
      tolerance=      5
      navigation_sources:
        gps
        end
        
''' % opts

   print putnav_cnt
   put = open(cntfile, 'w')
   put.write(putnav_cnt)
   put.close()
  
#--- end putnav -------------------------------------------------------

def aedit(opts):
   #write aedit_tmp.m and run it

   aed = open('aedit.m__','r')
   aed_list = aed.readlines()
   aed.close()

   for ii in range(0, len(aed_list)):
      aed_list[ii] = aed_list[ii].replace('__startdd__', str(opts['scn_startdd']))
      aed_list[ii] = aed_list[ii].replace('__enddd__', str(opts['loaded_enddd']))

   aedout = open('aedit_tmp.m','w')
   for ii in range(0, len(aed_list)):
      aedout.write(aed_list[ii])
   aedout.close()


   try:
      (runstr, output) = opts['M'].matlab('asetup; aflagit_setup; aedit_tmp')
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            

   opts['returnval'] = 1
   

#--- end aedit -------------------------------------------------------


def lst_temp(opts, cntfile):
   # list temperature
   
   lsttemp_cnt = '''
   dbname:         ../adcpdb/%(dbname)s
   output:           %(afilebase)s.tem
   step_size=        1
   year_base=        %(yearbase)d
   time_ranges:
   all

   ''' %opts
   
   print lsttemp_cnt
   ltemp = open(cntfile, 'w')
   ltemp.write(lsttemp_cnt)
   ltemp.close()
   
   opts['returnval'] = 1


#--- end lst_temp.py -------------------------------------------------------

def lst_btrk(opts, cntfile):
# Routine to generate a *.btm file using lst_btrk

  btmtrk_cnt = '''
     dbname:     ../../adcpdb/%(dbname)s
     output:     %(afilebase)s.btm
     step_size=  1
     year_base=  %(yearbase)s
     time_ranges:
  %(loaded_time_range)s         /* whole time range */

''' %opts
  print btmtrk_cnt
  btrk = open(cntfile, 'w')
  btrk.write(btmtrk_cnt)
  btrk.close()

# --- end lst_btrk -------------------------------------------------------

def refabsbt(opts, cntfile):
   # Routine to run refabsbt on bottom track data
   ref_cnt = '''
      fix_file_type:    simple
      reference_file:   %(afilebase)s.btm
      fix_file:         ../../nav/%(fixfile)s
      output:           %(afilebase)s.ref
      year_base=        %(yearbase)s
      ensemble_length=  %(ens_len)s
      gap_tolerance=    60
      
''' % opts


   print ref_cnt
   ref = open(cntfile, 'w')
   ref.write(ref_cnt)
   ref.close()


#--- end refabsbt -------------------------------------------------------

def btcaluv(opts):
   # use output from lst_btrk and refabsbt to calculate bototm track amp and phase

   matstr = '''


   %(displaystr)s
   
   [a, ph, gd_mask] = ...
      btcaluv('%(afilebase)s.ref','%(afilebase)s.btm',...
          'ADCP %(cruisename)s step size %(btrk_step)d ', %(btrk_step)d);
          
''' % opts

   if opts['logfile']:
      logw = open(opts['logfile'], 'a')
      logw.write('about to run\n' + matstr + '\n')
      logw.close()

      
   mat = open('callbtcaluv_tmp.m', 'w')
   mat.write(matstr)
   mat.close()



   try:
      (runstr, output) = opts['M'].matlab('btcaluv')
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            


#--- end btcaluv -------------------------------------------------------


def timslip(opts, cntfile):

   opts['ref_l0'] = 1;
   opts['ref_l1'] = (opts['wtrk_step'] -3)/2
   opts['ref_r0'] = opts['ref_l1'] + 3
   opts['ref_r1'] = opts['wtrk_step'] -1


   tslip_cnt = '''
   fix_file_type:      simple
   fix_file:           ../../nav/%(fixfile)s
   
   reference_file:     ../../nav/%(afilebase)s.nav
   output_file:        %(afilebase)s_%(wtrk_step)s.cal
   
   year_base=          %(yearbase)d
   min_n_fixes=        %(wtrk_step)s /* 5 7 9 */

   n_refs=             %(wtrk_step)s /* 5 7 9 */
   
   i_ref_l0=           %(ref_l0)s
   i_ref_l1=           %(ref_l1)s /* 1 2 3 */
   i_ref_r0=           %(ref_r0)s /* 4 5 6 */
   i_ref_r1=           %(ref_r1)s /* 4 6 8 */
   
   up_thresh=          3.0        /* m/s */
   down_thresh=        3.0        /* m/s */
   turn_speed=         2.0        /* m/s */
   turn_thresh=        60         /* degrees */
   
   dtmax=              360        /* seconds, for 300-second ensembles */
   tolerance=          5.e-5      /* days, about 5 seconds */
   grid:               ensemble
   
   use_shifted_times?  no

''' % opts


   print tslip_cnt
   ref = open(cntfile, 'w')
   ref.write(tslip_cnt)
   ref.close()

   del opts['ref_l0']
   del opts['ref_l1']
   del opts['ref_r0']
   del opts['ref_r1']
            

#--- end of timslip -------------------------------------------------------


def adcpcal(opts):
   # use output from timeslip  to calculate bototm track amp and phase

   matstr = '''

   %(displaystr)s
   
    adcpcal('%(afilebase)s_%(wtrk_step)s.cal', 'ADCP %(cruisename)s %(wtrk_step)sens')
    
''' % opts


   if opts['logfile']:
      logw = open(opts['logfile'], 'a')
      logw.write('about to run\n' + matstr + '\n')
      logw.close()

   mat = open('calladcpcal_tmp.m', 'w')
   mat.write(matstr)
   mat.close()



   try:
      (runstr, output) = opts['M'].matlab('calladcpcal_tmp', verbose=1)
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            


# --- end adcpcal -------------------------------------------------------

def plottemp(opts):


   try:
      (runstr, output) = opts['M'].matlab('plottemp')
   except:
      if opts['verbose']:
         logappend(opts['logfile'], (runstr, output))
            


#--- end plottemp -------------------------------------------------------

def timegrid(as_vars, cntfile):

   tmgrd_cnt = '''
   output: %(afilebase)s.tmg
   time_interval: %(timegrid)s
   time_range: %(loaded_time_range)s
   ''' %as_vars

   if (as_vars['timegrid'] == 0):
      as_vars['ss_string'] = 'single'
      #must use time range at bottom instead of file, don't run timegrid
      as_vars['timegridstring'] = as_vars['loaded_time_range']
      as_vars['run_timegrid'] = 0
   else:
      as_vars['ss_string'] = 'separate'
      #use timegrid file at the bottom, must run timegrid
      as_vars['timegridstring'] = '@%s.tmg' % (as_vars['afilebase'])
      run_timegrid = 1

   if run_timegrid:
      fid = open(cntfile, 'w')
      fid.write(tmgrd_cnt)                
      fid.close()

#--- end timegrid -------------------------------------------------------

def llgrid(as_vars, cntfile):             #### not yet implimented
   # make llgrid
   llgrid_cnt = '''

   dbname:        %(dbpath)s
   output:        %(afilebase).llg
   year_base:     %(yearbase)d
   step_size:     1
   lat_origin:    -0.05   /* center bins */
   lat_increment:  0.10   /* grid by 1/10 degree latitude */
   lon_origin:    -0.05
   lon_increment:  0.10   /* and 1/10 degree longitude */
   time_ranges:
   time_range: %(loaded_time_range)s
   
''' % opts
   
   fid = open(cntfile, 'w')
   fid.write(llgrid_cnt)
   fid.close()
   
#--- end llgrid -------------------------------------------------------

def as_vect(as_vars, cntfile):
   #adcpsect for vectors

   as_vars['dbpathname']  = os.path.join(as_vars['dbpath'], as_vars['dbname'])
   as_vars['outpathname'] = os.path.join(as_vars['outdir'], as_vars['vbasename'])

   asvect_cnt = '''
   dbname:             %(dbpathname)s
   output:             %(outpathname)s
   step_size:          1        /* must be 1 for navigation */
   ndepth:             %(numbins)d
   time_ranges:        %(ss_string)s        
   year_base=          %(yearbase)d

   option_list:
       pg_min=                %(pgmin)s
       reference:             final_ship_ref
       regrid:                average
          depth
          grid_list number=   %(grid_list_number)d
          boundaries:        %(firstdepth)d  %(deeper_boundaries)s
                             /* The 1st boundary must */
                             /* start at or below the */
                             /* shallowest depth bin  */
    vector:
      minimum_npts=        2
     flag_mask:        ALL_BITS
       end
     end                     /* one big time range (if "single") or if "separate", */
%(gridfilestring)s           /* explicit time ranges or of the form "@gridfile" */

''' % as_vars
   
   # do vector
   fid = open(cntfile, 'w')
   fid.write(asvect_cnt)
   fid.close()

   del as_vars['dbpathname'] 
   del as_vars['outpathname']

#--- end as_vec -------------------------------------------------------


def as_cont(as_vars, cntfile):
   #adcpsect for contours

   as_vars['dbpathname']  = os.path.join(as_vars['dbpath'], as_vars['dbname'])
   as_vars['outpathname'] = os.path.join(as_vars['outdir'], as_vars['cbasename'])

   ### need to fix grid number and increment

   ascont_cnt = '''
   dbname:             %(dbpathname)s
   output:             %(outpathname)s
   step_size:          1        /* must be 1 for navigation */
   ndepth:             %(numbins)d
   time_ranges:        %(ss_string)s        
   year_base=          %(yearbase)d

   option_list:
       pg_min=                %(pgmin)s
       reference:             final_ship_ref

       regrid:         average
                       depth

                       grid number=    %(gridnum)s
                       origin=         %(firstdepth)s
                       increment=      %(increment)s

                         
     flag_mask:        ALL_BITS
       end
     end
%(gridfilestring)s             /* explicit time ranges or of the form "@gridfile" */

''' % as_vars
   
   # do vector
   fid = open(cntfile, 'w')
   fid.write(ascont_cnt)
   fid.close()

   del as_vars['dbpathname'] 
   del as_vars['outpathname']

#--- end as_cont -------------------------------------------------------
