#!/usr/bin/python

#NOTE: you may need to change the path to python in the first line

'''
==========================================================================
======================== quick_adcp.py documentation =====================
==========================================================================

        run this                                to see this
    --------------                           ------------------
quick_adcp.py --help               :        (this page)
                                          :
quick_adcp.py --overview           : introduction to quick_adcp.py
                                          :      processing steps
                                          :
                                          :
                                          :
quick_adcp.py --howto_ping         : HOWTO for pingdata
                                          :
quick_adcp.py --tips               : tips for new users
      
                                          :
quick_adcp.py --varvals            : print variables and current values
quick_adcp.py --vardoc             : print documentation for variables
                                          :
                                          :      Simple example for:
                                          :      -------------------
quick_adcp.py --commands  LTA      :        LTA or ENS files (averaged)
quick_adcp.py --commands  ENX      :        ENX files (earth coords)
quick_adcp.py --commands  ENS      :        ENS files (beam coords)
quick_adcp.py --commands  UHDAS    :        UHDAS files
quick_adcp.py --commands  pingdata :        pingdata files
quick_adcp.py --commands  delblks  :        UHDAS "incremental" (at sea)
quick_adcp.py --commands  lastfiles:        LTA or pingdata "incremental"


NOTE to Windows users: replace 'quick_adcp.py' with 'quickadcp_py' and
                       replace 'adcptree.py' with 'adcptree_py'.  These
                       are .BAT files that set enviromennt variables and
                       call the respective python versions.

NOTES:
- All these switches use TWO dashes attached to a word. If you use one
  dash or leave a space afterwards, it will fail.
- DON\'T FORGET to quote wildcards: "pingdata.???" or "*.LTA"
- There is a useful little perl script called "linkping.prl" which might
  help in getting pingdata files named sequentially if your cruise has
  multiple legs.
- APOLOGIES:
  (1) some matlab routines do not display feedback to the screen
      (at the moment).  Read  the messages to see where to look
      for more information.
  (2) Only pingdata and UHDAS have automatic generation of a  heading
      correction (from a gps-aided defice such has Ashtech).  It\'s coming.
==========================================================================

'''


## Jules Hummon; 2001/01/07, from lastfiles.py
## moved as_all to fall after putnav; added -readme
## JH 2002/01/24 added refsm (still using smoothr for ref layer plots)
##               changed switches to --help, --overview, --howto
## lastfiles.py ported to python by C. Flagg
## JH 2002/08/31 merge lastfiles.prl, quick_adcp.prl, lastfiles.py

## JH 2004/06/01 - adding posmv heading correction, uhdas incremental, update gbin
##
##      
##      new timerange convention:
##          scn_time_range        correct times from scanned files
##          origdb_time_range     time range from database at start 
##          remaining_time_range  time range after blkfile deletion (incremental)
##          thisload_time_range   time range from files to be loaded 
##          loaded_time_range     full db time range after load (same as 'all')
##     ------------
##     specifically: for batch mode, any kind
##        origdb = [long ago]
##        remaining_time_range (unused?)
##        scn_time_range = thisload_time_range = loaded_time_range
##     ------------
##     incremental PING and LTA: beginning of file implies blk file boundary
##       adcpdb blocks: | blk0 | blk1 | blk2 |  blk3 |  blk4 | blk5 | (unloaded)
##          files (eg.) | 0      1      2       3       3      4    |  4   |  5    
##
##        'lastfiles = 2' means 
##           "delete blockfiles from files 3 and 4 (delete blk5, blk4, blk3)"
##        origdb_time_range:    beginning of file0 to file4 (at the time of load)
##        scn_timerange:        file 3,4,5 at the time of scan
##        remaining_time_range: file 0,1,2
##        thisload_time_range:  file 3,4,5
##        loaded_time_range:    file 0,1,2,3,4,5
##
##      algorigm: everything is driven by the filelist[last] and the repeated
##          "deletion of blockfiles + lstblk" until enough is deleted.
##          Relies on "start of file" implying "start of block"
##      
##     ------------
##     incremental UHDAS
##       adcpdb blocks: | blk0      |  blk1 |   blk2               | blk3
##          files (eg.) | 0,1,2     |  3    | 4,5,6 (first part)   | 6,7,8
##        'delblks = 1' means "delete 1  blockfile"
##        (delblks = 0  means "assume we're starting a blk now" (for debugging)
##        origdb_time_range:    beginning of file0 to file4 (at the time of load)
##        scn_timerange:        scan all files
##        remaining_time_range: blk 0,1,2
##        thisload_time_range:  beginning of blk3 through end of scn_time_range
##        loaded_time_range:    all
##        


import sys, os, string, re, glob, shutil, getopt, time
from types import *
from quick_subs import *
from quick_user import *
from quick_docs import *
from yn_query import yn_query
from mfile import m_to_dict


#----------------------------------------------------------------------


def usage():
   print __doc__
   sys.exit()

#----------------------------------------------------------------------

def print_vals(opts, headerstr=None):

   strlist = []

   if headerstr:
      strlist.append(headerstr)


   keys = opts.keys()           ## need to do this in two lines:
   keys.sort()                  ## (1) get keys, (2) sort it

   for key in keys:
      s = '%s   %s' % (string.ljust(key, 30), str(opts[key]))
      strlist.append(s)


   s = '\n\n'
   strlist.append(s)
   return string.join(strlist, '\n')


#----------------------------------------------------------------------

binos_msg1 = '''
I can't figure out where your codas3 executables are.
Your sys.platform is: %s.
It is not one of the platforms known to this program, which are:
   %s
Please use the --binos option when calling quick_adcp.py.
'''

def opt_paths(opts):
   if opts['progdir'] == '':  # try to deduce it
      fullname = sys.argv[0]
      if not os.path.abspath(fullname):
         print 'must set option "progdir" explictly'
         print 'this is not supposed to happen.'
         print 'Please let me know what platform this is so I can fix it.'
         print '     Thanks, Jules    (hummon@hawaii.edu)'
         sys.exit()

      ## Assuming $prog/codas3/bin/script/quick_adcp.py
      [scriptpath, quickname] = os.path.split(fullname)
      [binpath, scriptname] = os.path.split(scriptpath)
      [codaspath, binname] =  os.path.split(binpath)
      [progdir, codasname] =  os.path.split(codaspath)
      print 'progdir = ', progdir
      opts['progdir'] = progdir

   if opts['binos'] == '':
      d = {'win32':'mgw', 'darwin':'osx', 'sunos5':'sol', 'linux2':'lnx'}
      try:
         opts['binos'] = d[sys.platform]
         print 'binos = ', opts['binos']
      except KeyError:
         print binos_msg1 % (sys.platform, ', '.join(d.keys()))
         sys.exit()


   ### set the matlabpath string
   matlabpath = os.path.join(opts['progdir'], 'matlab')  #global
   matdirs = [os.path.join(matlabpath, 'mex5'),
              os.path.join(matlabpath, 'mex6'),
              os.path.join(matlabpath, 'utils'),
              os.path.join(matlabpath, 'misc'),
              os.path.join(matlabpath, 'codas3')]
   matdirs.append(os.path.join(matlabpath, 'rawadcp'))
   for subdir in  ('codasutils', 'logging', 'avg_edit', 'writeblk', 
                   'quality', 'utils', 'uhdas_plot'):
      matdirs.append(os.path.join(matlabpath, 'rawadcp', subdir))
   matdirs.append(os.path.join(matlabpath, 'rawadcp', 'sontek'))
   matdirs.append(os.path.join(matlabpath, 'rawadcp', 'rdi'))
   #opts['matpathstr'] = string.join(matdirs, ':')
   opts['matpathlist'] = matdirs

   return opts

#-----------------------------------------------------------------------

def check_opts(opts):



   if (opts['vardoc']):
      print_vardoc()
      sys.exit()


   ## determine instrument name, ping type, and data type
   opts['datatype'] =    opts['datatype'].lower()
   opts['instclass'] =   opts['instclass'].lower()
   opts['pingtype'] =    opts['pingtype'].lower()

   ## jump out if necessary
   if opts['datatype'] not in ('pingdata', 'ping', 'uhdas', 'lta', 'sta',
                               'enr', 'ens', 'enx'):
      print 'ERROR: data type "%s" not supported' % opts['datatype']
      sys.exit()

   ## test other combinations

   ## test uhdas requirements 
   if opts['datatype'] == 'uhdas':
      if opts['cfgbase'] == '':
         print 'ERROR: uhdas processing requires use of "cfgbase" files'
         sys.exit()
      else:
         cfgfile = os.path.join(opts['cfgpath'], '%s_cfg.m' % opts['cfgbase'])
         if not os.path.exists(cfgfile):
            print 'uhdas: must be able to find [cruiseid]_cfg.m'
            print 'cannot find it in %s' % cfgfile
            sys.exit()
         procfile = os.path.join(opts['cfgpath'], '%s_proc.m' % opts['cfgbase'])
         if not os.path.exists(procfile):
            print 'uhdas: must be able to find [cruiseid]_proc.m'
            print 'cannot find it in %s' % procfile
            sys.exit()
      ## must set these manually if adcptree.py did npt create instping.m
      if not os.path.exists(os.path.join(opts['cfgpath'], 'instping.m')):
         print 'using "uhdas", cannot find "instping.m"'
         if opts['instclass'] == 'unset':
            print 'cannot find "instclass" in options.  must set that first'
            sys.exit()
         if opts['pingtype'] == 'unset':
            print 'cannot find "pingtype" in options.  must set that first'
            sys.exit()
         if opts['instname'] == 'unset':
            print 'cannot find "pingtype" in options.  must set that first'
            sys.exit()
            
      

   ## get ping type
   if opts['datatype'] == 'pingdata':
      opts['pingtype'] = 'nb';
      opts['instclass'] = 'nb';
   elif opts['datatype'] in ('sta','lta'):
      opts['pingtype'] = 'first';

   ## check pingtype
   if (opts['datatype'] == 'uhdas'):
      print 'processing data logged with uhdas: '
      print '  instclass, instname, and pingtype were set by adcptree.py'
      print '  look in  config/instping.m for values'
   else:
      if opts['pingtype'] not in ('nb', 'bb', 'son', 'first'):
         print 'ERROR: ping type "%s" not supported' % opts['pingtype']
         sys.exit()
      if opts['instclass'] not in ('os', 'bb', 'nb', 'son'):
         print 'ERROR: instrument type "%s" not supported' % opts['instclass']
         sys.exit()
         

   ## no file specified but heading correction desired
   if opts['datatype'] in ('pingdata', 'ping'):
      if (opts['no_head_corr'] == 0):
         if (opts['rotate_file'] == ''):
            opts['time_angle_file'] = '%s.ang' % opts['dbname']
         else:
            opts['time_angle_file'] = opts['rotate_file']
   elif (opts['datatype'] == 'uhdas') and (opts['head_func'] != ''):
      if (opts['rotate_file'] == ''):
         opts['time_angle_file'] = '%s.ang' % opts['dbname']
      else:
         opts['time_angle_file'] = opts['rotate_file']
   else:
      ## no heading correction routine available yet, but can use a file
      opts['no_head_corr'] == 1
      opts['head_func'] == ''
      if (opts['rotate_file'] == ''):
         opts['time_angle_file'] = '%s.ang' % opts['dbname']
      else:
         opts['time_angle_file'] = opts['rotate_file']


   if opts['datatype'] in ('ens', 'enx'):
      if opts['lastfiles'] != 0 or opts['delblks'] != []:
         print 'ERROR: cannot do incremental processing with VmDAS data'
         sys.exit()
         
      if opts['ducer_depth'] == -1:
         print 'ERROR: must set transducer depth for single-ping VmDAS data'
         sys.exit()
   else:
      opts['ducer_depth'] = ''
      if opts['datatype'] in ('sta','lta'):
         print 'transducer depth is set in the data files.'
         print 'Use fix_dpth if necessary AFTER loading'
      elif opts['datatype'] == 'uhdas':
         print 'transducer depth set in *_cfg.m file\n'

   if (opts['datatype'] == 'ens'):
      if opts['ens_halign'] == 'None':
         print 'ERROR: must set heading offset for ENS files'
         sys.exit()
      else:
      ## assuming it's a number
         opts['ens_halign'] = float(opts['ens_halign'])


   ## test incremental type
   if opts['lastfiles'] != 0:
      if opts['datatype'] not in ('lta', 'sta', 'pingdata'):
         print 'ERROR: "lastfiles" can only be used with the following:'
         print '    pingdata, lta, or sta'
         sys.exit()
   elif opts['delblks'] != []:
      if opts['datatype'] != 'uhdas':
         print 'ERROR: "lastfiles" can only be used with uhdas data:'
         sys.exit()
   

   ## set minimum PG to default if not already set
   if opts['pgmin'] == 0:
      if  opts['datatype'][-3:] in ('ens', 'ENS'):
         opts['pgmin'] = 20
      elif  opts['datatype'][-2:] in ('uh',):
         opts['pgmin'] = 20
      else:
         opts['pgmin'] = 50

   ## string for globbing on adcp data files
         ## (for raw, filelist passed to matlab)
   if opts['datafile_glob'] == None:
      if  opts['datatype'] == 'enr':
         opts['datafile_glob'] = '*.ENR'
      if  opts['datatype'] == 'ens':
         opts['datafile_glob'] = '*.ENS'
      if  opts['datatype'] == 'enx':
         opts['datafile_glob'] = '*.ENX'
      if  opts['datatype'] == 'sta':
         opts['datafile_glob'] = '*.STA'
      if  opts['datatype'] == 'lta':
         opts['datafile_glob'] = '*.LTA'
      if  opts['datatype'] == 'pingdata':
         opts['datafile_glob'] = 'PINGDATA.???'
      if  opts['datatype'] == 'uhdas':
         opts['datafile_glob'] = '*.raw'
   ## set path to data directory
   if opts['datadir'] == None:
         opts['datadir'] = os.path.join('..','ping')

   if opts['datatype'] == 'uhdas':
      opts['cfg_fullpath'] = os.path.abspath(os.path.join(opts['cfgpath']))
   else:
      opts['cfg_fullpath'] = ''


   ## check to make sure datatype and instrument match
   if opts['instclass'] == 'os':
      if opts['pingtype'] not in ('nb','bb', 'first'):
         print 'ERROR: must choose nb or bb ping type for Ocean Surveyor'
         print 'ping type %s not allowed' % (opts['pingtype'])
         sys.exit()
   else:
      opts['pingtype'] = opts['instclass']


  # print documentation
   if opts['setup_info']:
      print_setup_info()
      sys.exit()

   if opts['overview']:
      print_overview()
      sys.exit()

   if opts['tips']:
      print_tips()
      sys.exit()

   if opts['howto_ping']:
      print_howto_ping()
      sys.exit()

   if opts['commands'] != '':
      opts['commands'] = opts['commands'].lower()
      if opts['commands'] in  ('lta', 'sta'):
         print_lta_commands()
         sys.exit()
      elif opts['commands'] in ('ens', 'enx'):
         print_enx_commands()
         sys.exit()
      elif opts['commands'] in ('uhdas',):
         print_uhdas_commands()
         sys.exit()
      elif opts['commands'] in ('pingdata', 'ping'):
         print_pingdata_commands()
         sys.exit()
      elif opts['commands'] in ('lastfiles',):
         print_lastfiles_commands()
         sys.exit()
      elif opts['commands'] in ('blkfiles',):
         print_blkfiles_commands()
         sys.exit()
      elif opts['commands'] in ('autoedit',):
         print_autoedit_commands()
         sys.exit()

   if opts['max_BLKprofiles'] > 399:
      print 'max_BLKprofiles must be less than 400.  resetting to 399\n'
      opts['max_BLKprofiles'] = 399

   if opts['lastfiles'] != 0 or opts['delblks'] != []:
      opts['allow_reload'] = 1
   else:
      opts['allow_reload'] = 0
      

   if opts['hidedisplay'] == 1:
      opts['displaystr'] = '''
      set(0,'defaultfigureposition',[0 0 1 1])
      set(0,'defaultfiguremenubar','none')    
      set(0,'defaultfiguretoolbar','none')
      '''
   else:
      opts['displaystr'] = ''' %% no change to display %% '''


   return opts

#----------------------------------------------------------------------

def parse_opts(argv = None):
   import getopt

   opts = {}

   if not argv:


      ## defaults for input variables

      opts['help'] = 0             # print help

      opts['howto_ping'] = 0       # print a HOWTO for pingdata
      opts['howto_lta'] = 0        # print a HOWTO for LTA
      opts['varvals'] = 0          # print dictionary values
      opts['vardoc']  = 0          # detailed use for each variable
      opts['overview'] = 0         # print overview help

      opts['commands'] = ''        # print specific help
      opts['setup_info'] = 0       # print help for setting up
      opts['tips'] = 0             # print tips

      opts['verbose'] = 0          # turn on error messages.  This includes
                                   #   writinng messages to the screen
                                   #   and appending the same error messages
                                   #   to the log file.

      opts['cntfile'] = None       # override defaults with values from this
                                   #   control file.  Format is the same as
                                   #   command line options, but whitespace is
                                   #   unimportant.  Do not quote wildcard
                                   #   characters in the cnt file (whereas
                                   #   you would for the command line)
                                   # Defaults are overridden by this cntfile
                                   # Commandline options can override either.

      ### these three do not have useful defaults and are REQUIRED for use
      opts['yearbase'] = None      # 4 digits
      opts['dbname']   = None      # limit 5 characters, start with "a"
                                   #   then append  4 characters
      ####
      opts['instclass'] = 'unset'  # instrument category, describes instrument
                                   #       Default is "nb"
                                   #      choose from:
                                   #      os    (phased array "ocean surveyor")
                                   #      bb    (broadband instrument)
                                   #      nb    (narrowband instrument)
                                   #      son   (sontek)

      opts['pingtype'] = 'unset'   #  only required for instrument "os"
                                   #   (then choose "nb" or "bb")
                                   #  For all others, only one pingtype is
                                   #     possible ( therefore pingtype defaults
                                   #     to instrument type)

      opts['datatype'] = 'pingdata'
                                   # choose what kind of data logging (and
                                   #   hence which kind of data files
                                   #   are being processed)
                                   #
                                   # default is "pingdata"
                                   #
                                   #  name          what
                                   #  ----          ----
                                   #  "pingdata"     implies DAS2.48 (or 2.49) NB
                                   #                     data, already averaged
                                   #  "uhdas"        implies known directory
                                   #                     structure for access
                                   #                     to raw data
                                   #  "lta", "sta"   VmDAS averages
                                   #  "enx"          implies VmDAS single-ping
                                   #                     data,time-corrected,
                                   #                     navigated with attitude

      opts['instname'] = None      # instrument directory name; uhdas only. This
                                   #  is the name of the directory with the raw
                                   #  data and hence the gbin data.  EG. "os38"

      opts['procdir'] = os.path.basename(os.getcwd()) #processing directory name

      opts['cruisename'] = os.path.basename(os.getcwd()) #title name only
                                   # defaults to the averaged data case, where
                                   # cruise name is the same as procdir name

      opts['datadir'] = None       # data path. defaults are:
                                   #     (1) relative to scan or load directory
                                   #     (2) full path
                                   #  ../ping           (for LTA, ENS, pingdata)
                                   #  uhdas_dirbase/raw/instname (for uhdas)

      opts['datafile_glob'] = None #look for sequential data files. defaults are:
                                   #  pingdata.???  (for pingdata)
                                   #  *.LTA         (for LTA files)
                                   #  *.ENS         (for ENS files)
                                   #  *.raw         (for uhdas)


      opts['fixfile'] = None       # override the default fix file. defaults are:
                                   # if datatype is pingdata, default is
                                   # [dbname].ags (for pingdata)
                                   # [dbname].gps (for LTA, ENS, or uhraw)

      opts['ub_type']  = '1920'    # For NB DAS only: User buffer type:
                                   # 720 for demo; 1920 for recent cruises

      opts['ens_len'] = 300        # seconds in an ensemble

                                   # The next two variables cause the matlab
                                   # program "xducerxy" to run, creating
                                   # a new fix file (dbname.agt).  These
                                   # values must be specified any time the fix
                                   # file is used, including steps2rerun
                                   #
      opts['xducer_dx'] = 0        # athwartship distance between transducer
                                   # and gps antenna: positive if transducer
                                   # is starboard of the antenna.
                                   # This option is usually not necessary
                                   #
      opts['xducer_dy'] = 0        # fore/aft distance between transducer
                                   # and gps antenna: positive if transducer
                                   # is forward of the antenna.
                                   # This option is usually not necessary


                                   ##  choose one or the other for smoothed nav
      opts['use_refsm'] = 0        # use refsm
      opts['use_smoothr'] = 0      # use adcpsect, refabs, smoothr
                                   # NOTE: runs adcpsec, refabs, smoothr
                                   # anyway for reflayer plots and watertrack

      opts['rotate_amplitude'] = 1 # start with 1.0 for amp and 0.0 for phase
      opts['rotate_angle']     = 0 # transducer offset (from calibrations)
                                   #    (apply this constant value)

      opts['rotate_file']     = '' # transducer offset in a file, eg. ash-gyro
                                   #    (requires an existing file;
                                   # REQUIRES one angle correction per ensemble
                                   #     with the correct times for ensembles

      opts['no_head_corr'] = 0     # default: yes, there is heading correction
                                   # from a gps heading source such as Ashtech
                                   # use '--no_head_corr' to turn that off
                                   # PINGDATA ONLY

      
      opts['rl_startbin'] = 2      # first bin for reference layer
      opts['rl_endbin'] = 20       # last bin for reference layer

      opts['pgmin'] = 0            # only accept PG greater than this value
                                   # --> don't go below 30 <--
                                   # default is 50 for pre-averaged datatypes
                                   #      ('pingdata', 'lta', and 'sta')
                                   # default is 20 for single-ping datatypes
                                   #   ('uhdas', 'enr', 'ens', 'enx')

      opts['numbins'] = 100        # max number of bins to extract for editing

      opts['progdir'] = ''         #where to find the programs

      opts['binos'] = ''           #subdirectory name  with executables

      opts['firstdepth'] = 25      #shallowest depth bin for matlab calculations



      opts['nozap'] = 0            #make a "setup.m" suitable for zap editing
                                   # default is false (i.e. DO setup for 'zap')
      opts['auto'] = 0             # 1: whiz along (do not prompt); do all
                                   #    requested steps (default is all step)
                                   #    0 to prompt at each step

      opts['steps2rerun'] = ''     # colon-separated list of the following:
                                   # 'rotate:apply_edit:navsteps:calib:matfiles'
                                   # designed for batch mode; assumes codasdb
                                   #  already in place, operates on entire
                                   #  database.
                                   #
                                   # 'rotate' is:
                                   #   apply amplitude and phase corrections
                                   #   using these (if specified):
                                   #      rotate_angle (contstant angle)
                                   #      rotate_amplitude (scale factor)
                                   # Does NOT support time-dependent correction
                                   #
                                   # 'navsteps' is:
                                   #   [adcpsect+refabs+smoothr] for ref plots
                                   #    and refsm (if specified)
                                   #    and putnav (using specified navigation)
                                   #
                                   # 'apply_edit' is:
                                   #   badbin, dbupdate, set_lgb, setflags
                                   #
                                   # 'calib' is:
                                   #   botmtrk (refabsbt, btcaluv)
                                   #   watertrk (adcpsect, timslip, adcpcal)
                                   #
                                   # 'matfiles' is:
                                   #   adcpsect for vector and contour
                                   #   (specify firstdepth for better results)

      ############### for single-ping proccessing only ################

      opts['cfgpath'] = 'config'   # UHDAS only
                                   # Find cfg files (*_cfg.m, *_proc.m) here.

      opts['cfgbase'] = ''         # UHDAS only
                                   #   path and processing information
                                   #   (file name prefix only.)


      opts['ducer_depth'] = -1     # for VmDAS processing, need to
                                   # specify transducer depth -- REQUIRED

      opts['ens_halign'] = 'None'  # for ENS processing only, conversion from
                                   # beam to earth coordinates: need to
                                   # specify transducer alignment -- REQUIRED
                                   # (ENX files are already in earth coordinates)
                                   # (see 'rotate_angle' and 'rotate_file'
                                   #    for rotation of codas database
                                   #    as part of calibration steps

      opts['enddday_file'] = None  # single-ping processing only: use this
                                   #   file for ensemble dday delimiters; gets
                                   #   copied to directory as ensemble.enddday 
                                   #   Dday should be first column, (as lst_hdg
                                   #    or lst_temp output from other database)

      opts['head_func'] = 'ashrot' # use this python function to get the
                                   # correction.  'ashrot' is inside
                                   # quick_adcp.py -- if something else
                                   # is specified, it is imported from
                                   # the PYTHONPATH and run in the same
                                   # manner as ashrot, from within cal/rotate.
                                   # (see quick_user.py for an example;
                                   # use "--head_func get_posmvdh"

      opts['rbindir'] = '/home/adcp/cruise/rbin'  ### UHDAS ONLY 
                                   # location of rbin instrument directories,
                                   #   i.e. root of gyro, adu2, posmv, etc.
                                   #   default: /home/adcp/cruise/rbin

      opts['update_gbin'] = 0      # UHDAS ONLY: update remaining gbin files


      ################### for incremental processing only ####################


      opts['hidedisplay'] = 0      # show figures as they are made (turn off
                                   # to run from a script with cron on a ship

      opts['lastfiles'] = 0        # for incremental processing of pre-averaged
                                   # data (i.e. STA, LTA, or pingdata) only:
                                   # delete sufficient blocks from the database
                                   # to account for N data files. Reload the files
                                   
      opts['delblks'] = []         # for incremental processing of UHDAS data:
                                   # delete this nnumber of blocks from the database.
                                   # Averaging will continue with whatever is new.
                                   # Reload unloaded ens_blk*.cmd files.
                                   # usual usage: --delblks 1  (standard incremental)
                                   # debugging only: --delblks 0 (WILL reload)
                                   
      opts['find_pflags'] = 0      #  automatically find profile flags
                                   #  apply editing (dbupdate, etc)

      opts['max_BLKprofiles'] = 399# max number of profiles per block.  Might
                                   #  want to make smaller to test incremental
                                   #  processing (must be 399 or less)

      ## end of dictionary (defaults) definition


   else:

      ## read commandline
      try:
         options, args = getopt.getopt(argv, '',
                                       ['help',
                                        'howto_ping',
                                        'howto_lta',
                                        'varvals',
                                        'verbose',
                                        'tips',
                                        'commands=',
                                        'vardoc',
                                        'overview',
                                        'setup_info',
                                        'cntfile=',
                                        'yearbase=',
                                        'dbname=',
                                        'instclass=',
                                        'pingtype=',
                                        'datatype=',
                                        'instname=',
                                        'procdir=',
                                        'cruisename=',
                                        'datadir=',
                                        'datafile_glob=',
                                        'fixfile=',
                                        'ub_type=',
                                        'ens_len=',
                                        'xducer_dx=',
                                        'xducer_dy=',
                                        'use_refsm',
                                        'use_smoothr',
                                        'rotate_amplitude=',
                                        'rotate_angle=',
                                        'rotate_file=',
                                        'no_head_corr',
                                        'head_func=',
                                        'rbindir=',
                                        'update_gbin',
                                        'rl_startbin=',
                                        'rl_endbin=',
                                        'pgmin=',
                                        'numbins=',
                                        'progdir=',
                                        'binos=',
                                        'firstdepth=',
                                        'nozap',
                                        'auto',
                                        'steps2rerun=',
                                        # for single-ping
                                        'cfgbase=',
                                        'cfgpath=',
                                        'ducer_depth=',
                                        'ens_halign=',
                                        'enddday_file=',
                                        # for unattended automatic processing
                                        'hidedisplay',
                                        'lastfiles=',
                                        'delblks=',
                                        'max_BLKprofiles=',
                                        'find_pflags'])
         #all the rest are data file names
         opts['filelist'] = args
         if len(args) > 0:  #if filenames specified on command line, override glob
            opts['datafile_glob'] = ''

      except:
         print_vals(opts)
         print '\n -----> incorrect argument in argument list <----'
         print __doc__
         raise

      for o, a in options:
         if o in ('--help', ):
            print 'YOU INVOKED HELP'
            usage();
         elif o in ('--howto_ping', ):       opts['howto_ping']      = 1
         elif o in ('--howto_lta', ):        opts['howto_lta']       = 1
         elif o in ('--varvals', ):          opts['varvals']         = 1
         elif o in ('--overview', ):         opts['overview']        = 1
         elif o in ('--vardoc', ):           opts['vardoc']          = 1
         elif o in ('--commands', ):         opts['commands']        = a
         elif o in ('--setup_info', ):       opts['setup_info']      = 1
         elif o in ('--tips', ):             opts['tips']            = 1
         elif o in ('--verbose', ):          opts['verbose']         = 1
         elif o in ('--cntfile', ):          opts['cntfile']         = a
         elif o in ('--yearbase', ):         opts['yearbase']        = int(a)
         elif o in ('--dbname', ):           opts['dbname']          = a
         elif o in ('--instclass', ):        opts['instclass']       = a
         elif o in ('--pingtype', ):         opts['pingtype']        = a
         elif o in ('--datatype', ):         opts['datatype']        = a
         elif o in ('--instname', ):         opts['instname']        = a
         elif o in ('--procdir', ):          opts['procdir']         = a
         elif o in ('--cruisename', ):       opts['cruisename']      = a
         elif o in ('--datadir', ):          opts['datadir']         = a
         elif o in ('--datafile_glob', ):    opts['datafile_glob']   = a
         elif o in ('--fixfile', ):          opts['fixfile']         = a
         elif o in ('--ub_type', ):          opts['ub_type']         = str(a)
         elif o in ('--ens_len', ):          opts['ens_len']         = int(a)
         elif o in ('--xducer_dx', ):        opts['xducer_dx']       = int(a)
         elif o in ('--xducer_dy', ):        opts['xducer_dy']       = int(a)
         elif o in ('--use_refsm', ):        opts['use_refsm']       = 1
         elif o in ('--use_smoothr', ):      opts['use_smoothr']     = 1
         elif o in ('--rotate_amplitude', ): opts['rotate_amplitude']= float(a)
         elif o in ('--rotate_angle', ):     opts['rotate_angle']    = float(a)
         elif o in ('--rotate_file', ):      opts['rotate_file']     = float(a)
         elif o in ('--no_head_corr', ):     opts['no_head_corr']    = 1
         elif o in ('--rl_startbin', ):      opts['rl_startbin']     = int(a)
         elif o in ('--rl_endbin', ):        opts['rl_endbin']       = int(a)
         elif o in ('--pgmin', ):            opts['pgmin']           = int(a)
         elif o in ('--numbins', ):          opts['numbins']         = int(a)
         elif o in ('--progdir', ):          opts['progdir']         = a
         elif o in ('--binos', ):            opts['binos']           = a
         elif o in ('--firstdepth', ):       opts['firstdepth']      = int(a)
         elif o in ('--nozap', ):            opts['nozap']           = 0
         elif o in ('--auto', ):             opts['auto']            = 1
         elif o in ('--steps2rerun', ):      opts['steps2rerun']     = a
         # for single-ping
         elif o in ('--cfgbase', ):          opts['cfgbase']         = a
         elif o in ('--cfgpath', ):          opts['cfgpath']         = a
         elif o in ('--ducer_depth', ):      opts['ducer_depth']     = int(a)
         elif o in ('--ens_halign', ):       opts['ens_halign']      = a
         elif o in ('--enddday_file', ):     opts['enddday_file']    = a
         elif o in ('--head_func', ):        opts['head_func']       = a
         elif o in ('--rbindir', ):          opts['rbindir']         = a
         elif o in ('--update_gbin', ):       opts['update_gbin']      = 1
         # for unattended automatic processing
         elif o in ('--hidedisplay', ):      opts['hidedisplay']     = 1
         elif o in ('--lastfiles', ):        opts['lastfiles']       = int(a)
         elif o in ('--delblks', ):          opts['delblks']         = int(a)
         elif o in ('--max_BLKprofiles', ):  opts['max_BLKprofiles'] = int(a)
         elif o in ('--find_pflags', ):      opts['find_pflags']     = 1
         else:
            print_vals()
            print 'switch <%s> failed' % o
            sys.exit()
      ## end of dictionary update from argv

   try:
      if opts['verbose']:
         print '\nargv: ', argv
         print '\noptions: ', options
         print '\nargs: ', args
         print '\nopts: ', opts
   except:
      pass

   return opts

#--- end parse_opts ------------------------------------------------------

# send in default options, return default values for adcpsect subroutines
# askw is "adcpsect key+word"

def get_asvars(opts):

   ## these eare reasonable defaults for nb150
   as_vars = {'dbname'        :    opts['dbname'],
              'dbpath'        :    '../adcpdb',
              'outdir'        :    '.',
              'cbasename'     :    'contour',
              'vbasename'     :    'vector',
              'afilebase'     :    'as_nav',
              'timegrid'      :    60,       ## 60 minutes,
              'loaded_time_range'    :    'all',
              'numbins'       :    opts['numbins'],
              'ss_string'     :    'separate',
              'rl_startbin'   :    opts['rl_startbin'],
              'rl_endbin'     :    opts['rl_endbin'],
              'yearbase'      :    opts['yearbase'],
              'pgmin'         :    opts['pgmin'],
              'firstdepth'    :    opts['firstdepth']}
   
   as_vars['increment'] = '10'
   as_vars['gridnum'] = '40'
   as_vars['deeper_boundaries'] = '50 75 125 175 225 275 325 375 425'
   as_vars['grid_list_number'] = 1 + len(as_vars['deeper_boundaries'].split())
   
   
   ## set some defaults for Kilo Moana
   if opts['instname'] == 'os38' and opts['pingtype'] == 'bb':
      print ' using adcpsect defaults for Kilo Moana, bb pings'
      as_vars['firstdepth'] = 40
      as_vars['increment'] = '20'
      as_vars['gridnum'] = '40'
      as_vars['boundaries'] = '75 125 175 250 300 350 400 450 500 550 600 650 700 750 800 850 900'
      as_vars['grid_list_number'] = 1 + len(as_vars['deeper_boundaries'].split())
      
   if opts['instname'] == 'os38' and opts['pingtype'] == 'nb':
      print ' using adcpsect defaults for Kilo Moana, nb pings'
      as_vars['firstdepth'] = 50
      as_vars['increment'] = '25'
      as_vars['gridnum'] = '70'
      as_vars['deeper_boundaries'] = \
             '100 150 250 350 450 550 650 750 850 950 1050 1150 1250 1350 1450 1550 1650'
      as_vars['grid_list_number'] = 1 + len(as_vars['deeper_boundaries'].split())
   
   return as_vars

#========================================================================
## (0) if sonuh or nbuh, set up raw path info
def run_rawsetup(opts):

   ## check about ensdday now
   if (opts['enddday_file'] != None):
      workdir =  'load'
      print 'ensemble time file is ' + opts['enddday_file']
      if not os.path.exists(opts['enddday_file']):
         print 'WARNING: file %s does not exist. Cannot use ensemble times' \
               %(opts['enddday_file'])
         if (opts['auto'] == 1):
            qa_default = 'y'
         else:
            qa_default = 'n'
            qa = yn_query('continue without ensemble time file?', 'ynq',
                          default=qa_default, auto=opts['auto'])
      # there was a file
      pqlist = def_qlist('copy file to load directory?', opts['auto'])
      pycmd = [copy_ensfile, (opts, )]
      runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])
   else:
      print 'no ensemble time file specified'
      if (opts['auto'] == 1):
         qa_default = 'y'
      else:
         qa_default = 'n'
      qa = yn_query('continue?', 'ynq',
                    default=qa_default, auto=opts['auto'])

#--- end run_rawsetup -------------------------------------------------------

## (0a) update gbin files
def run_updategbin(opts):


   if (opts['auto'] == 1):
      qa_default = 'y'
   else:
      qa_default = 'n'


   yn = yn_query('update gbin files now?' , 'ynq',
                    default=qa_default, auto=opts['auto'])

   if yn:
      print 'updating gbin files ...'
      pycmd = [update_gbin, (opts, )]
      runcmd(pq=yeslist(), pycmd=pycmd, logfile=opts['logfile'])


#--- end run_updategbin ---------------------------------------------------
def get_matcfgstr(opts):

   if opts['datatype'] == 'uhdas':
      if (opts['instclass'] != 'unset') and \
         (opts['pingtype']  != 'unset') and \
         (opts['instname']  != None):


         opts['matcfgstr']  = '''
         %% begin "matcfgstr" string, created by quick_adcp.py
         startdir = pwd;
         chdir('%(cfg_fullpath)s')
         instclass = %(instclass)s;
         pingtype = '%(pingtype)s';
         instname = '%(instname)s';
         %(cfgbase)s_cfg               %% sets up various directories
         %(cfgbase)s_proc              %% sets up parameters for processing
         chdir(startdir)
         %% end  "matcfgstr" string, created by quick_adcp.py
         ''' % opts   
         
      else:

         opts['matcfgstr']  = '''
         %% begin "matcfgstr" string, created by quick_adcp.py
         startdir = pwd;
         chdir('%(cfg_fullpath)s')
         instping;                     %% pingtype, instrument name, instclass
         %(cfgbase)s_cfg               %% sets up various directories
         %(cfgbase)s_proc              %% sets up parameters for processing
         chdir(startdir)
         %% end  "matcfgstr" string, created by quick_adcp.py
         ''' % opts
   else:

      opts['matcfgstr']  = '''
      %% begin "matcfgstr" string, created by quick_adcp.py
      startdir = pwd;
      instclass = %(instclass)s;
      pingtype = '%(pingtype)s';
      instname = '%(instname)s';
      yearbase = %(yearbase)s;
      cruiseid = '%(cruisename)s';
      %% end  "matcfgstr" string, created by quick_adcp.py
      ''' % opts            


   return opts

#--- end get_matcfgstr -------------------------------------------------------


## (1) scan data
def run_scandata(opts):


   trfile = '%(afilebase)s.tr' % opts
   timefile = '%(afilebase)s.scn' % opts

   print(flankline('step 1: scan data (create .scn file)'))

   startdir = os.getcwd()
   workdir =  'scan'
   os.chdir(workdir)

   if (opts['auto'] == 1):
      qa_default = 'y'
   else:
      qa_default = 'n'


   if opts['allow_reload']:
      yn = 1
   else:
      yn = yn_query('scan %s data now?' %opts['datatype'], 'ynq',
                    default=qa_default, auto=opts['auto'])

   if yn:
      print 'scanning files for time ranges...'
      if opts['datatype'] == 'pingdata':
         cntfile = 'scanping.tmp'
         file_msg(os.path.join(workdir, cntfile))
         pycmd = [scanping, (opts, cntfile)]
         syscmd = '%s %s' % (os.path.join(opts['bindir'], 'scanping'), cntfile)
         runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(), syscmd=syscmd,
                logfile=opts['logfile'])
      else:
         pycmd = [scanraw, (opts, )]
         runcmd(pq=yeslist(), pycmd=pycmd, logfile=opts['logfile'])

   os.chdir(startdir)

   #always get scan time range:
   # if scanning, remove it so it will be rewritten
   workdir = 'scan'
   try:
      os.remove(os.path.join(workdir,trfile))
   except:
      pass
   print 'writing time range from %s to %s' %(timefile, trfile)


#--- end run_scandata ---------------------------------------------------


## (2) load data
def run_loaddata(opts):
   isoverlapping = 0  #no overlap in existing db and new files (or none loaded yet)
   print(flankline('step 2: load data into codas database'))
   ## test if any of the new pingdata are already loaded


   if opts['datatype'] == 'uhdas':
      ## new paradigm -- try to finish making cmd files before finding time ranges
      # make load_uhblk_tmp.m and run it (creates .cmd, .bin, .gps1, .gps2)
      startdir = os.getcwd()
      workdir = 'load'
      pqlist =  def_qlist('make cmd and bin files for loading?', opts['auto'])
      pycmd = [loaduh, (opts, )]
      runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir,  logfile=opts['logfile'])


   # is there a database?: (needs dir.blk file and at least 1 data blk file)
   get_remaining_times(opts)

   ## block files and [dbname]dir.blk
   ## NOTE: overlapping(opts) is the function that deletes blk files if requested

   blkfiles = glob.glob(os.path.join('adcpdb', opts['dbname'] + '*.blk'))
   if len(blkfiles) > 1:
      isoverlapping = overlapping(opts)
   else:
      isoverlapping = 0;

   get_remaining_times(opts)
   get_data2load_timerange(opts) ## fills thisload_
   if opts['remaining_enddd'] >= opts['thisload_startdd']:
      print 'ERROR: trying to load data that REALLY IS already loaded'
      sys.exit()

   print 'using data files (as seen from scan/ or load/ directory):\n' + \
         string.join(opts['scanfilelist'], '\n') + '\n\n'


   if isoverlapping: # override
      loglist = ['\n', time.asctime()]
      ss = '**************************************************************\n' +\
           '************ some of these data are already loaded ***********\n' +\
           '**************************************************************\n' +\
           '\n\n ======> NOT LOADING NEW DATA <======= \n\n\n' +\
           'All further steps are performed on the original database\n\n'
      print ss; loglist.append(ss)
      logappend(opts['logfile'], loglist)
      opts['thisload_time_range'] = '';
      opts['this_startdd'] = None
      opts['this_enddd'] =   None

      yn = yn_query('continue processing?', 'yq', default='q',auto=opts['auto'])

   else:
      loglist = ['\n', time.asctime()]
      loglist.append('about to load %s files\n' %opts['datatype'])
      loglist.append(string.join(opts['scanfilelist'], '\n'))
      logappend(opts['logfile'], loglist)

      startdir = os.getcwd()
      workdir = 'load'
      if opts['datatype'] == 'pingdata':
         # make cntfile, use it
         cntfile = 'loadping.tmp'
         file_msg(os.path.join(workdir, cntfile))
         pycmd = [loadping, (opts, cntfile)]
         sqlist =  def_qlist('load pingdata now?', opts['auto'])
         syscmd = '%s %s' %(os.path.join(opts['bindir'],'loadping'), cntfile)
         runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd,
                workdir=workdir,  logfile=opts['logfile'])
      elif opts['datatype'] in ('lta', 'sta'):

         # make load_lta.m and run it (creates .cmd, .bin, .gps1, .gps2)
         pqlist =  def_qlist('make cmd and bin files for loading?',opts['auto'])
         pycmd = [loadlta, (opts,  )]
         runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir,logfile=opts['logfile'])

         # make ldcodas.cnt and run it
         cntfile = 'ldcodas.tmp'
         file_msg(os.path.join(workdir, cntfile))
         pycmd = [ldcodas, (opts, cntfile)]
         sqlist =  def_qlist('load averaged data now?', opts['auto'])
         syscmd = '%s %s' %(os.path.join(opts['bindir'],'ldcodas'), cntfile)
         runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd,
                workdir=workdir,  logfile=opts['logfile'])
      elif opts['datatype'] == 'uhdas':
         # make ldcodas.cnt and run it
         cntfile = 'ldcodas.tmp'
         file_msg(os.path.join(workdir, cntfile))
         pycmd = [ldcodas, (opts, cntfile)]
         sqlist =  def_qlist('load averaged data now?', opts['auto'])
         syscmd = '%s %s' %(os.path.join(opts['bindir'],'ldcodas'), cntfile)
         runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd,
                workdir=workdir,  logfile=opts['logfile'])
      else:
         print 'cannot deal with data %s' % opts['datatype']
         sys.exit()

      dbchanged = 1
      return dbchanged

#--- end run_loaddata -------------------------------------------------------


## lst config
def run_lstconfig(opts):
   workdir = 'adcpdb'
   cntfile = 'lst_conf.tmp'
   pycmd = [lst_conf, (opts, cntfile)]
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'lst_conf'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(),
          syscmd=syscmd, workdir=workdir,  logfile=opts['logfile'])


#--- end run_lstconfig --------------------------------------------


## (3) editsetup
def run_editsetup(opts):
   print(flankline('step 3 (create suitable setup files for editing)'))
   workdir =  'edit'
   if opts['nozap'] == 0: # i.e. yes, do zapsetup
      if not (os.path.exists(os.path.join(workdir, 'setup.m.demo'))):
         pqlist = def_qlist('make a new pre-configured setup.m ?', opts['auto'])
         pycmd = [zapsetup, (opts, )]
         runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir,
                logfile=opts['logfile'])

   ##  make asetup.m
   workdir =  'edit'
   if os.path.exists(os.path.join(workdir, 'asetup.m')):
      pqlist =  def_qlist('remake asetup.m ?', opts['auto'])
   else:
      pqlist =  yeslist()
   print 'making asetup.m'
   pycmd = [asetup, (opts, )]
   runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])

   ##  make aflagit_setup.m
   workdir =  'edit'
   if os.path.exists(os.path.join(workdir, 'aflagit_setup.m')):
      pqlist =  def_qlist('remake aflagit_setup.m ?', opts['auto'])
   else:
      pqlist = yeslist()
      print 'making aflagit_setup.m'
   pycmd = [aflagit_setup, (opts, )]
   runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])


#--- end run_editsetup -------------------------------------------------------



## (4) setflags
def run_setflags(opts):
   print(flankline('step 4: run setflags'))
   workdir = 'edit'
   cntfile = 'setflags.tmp'
   pycmd = [setflags, (opts, cntfile)]
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'setflags'), cntfile)
   sqlist =  def_qlist('run setflags?', opts['auto'])
   runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])


#--- end run_setflags --------------------------------------------


## (5) get navigation fixes
def run_getnav(opts):
   if opts['datatype'] == 'pingdata':
      print(flankline('step 5: get fixes from ubprint'))
      workdir = 'nav'
      cntfile = 'ubprint.tmp'
      file_msg(os.path.join(workdir, cntfile))
      pycmd = [ubprint, (opts, cntfile)]
      sqlist =  def_qlist('run ubprint?', opts['auto'])
      syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'ubprint'), cntfile)
      runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd,
             workdir=workdir, logfile=opts['logfile'])
   else: #running ldcodas
      print(flankline('step 5: get fixes from load/*.gps2 files'))
      workdir = 'nav'
      pqlist =  def_qlist('cat navigation?', opts['auto'])
      pycmd = [catnav, (opts, )]
      runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir,
             logfile=opts['logfile'])

#--- end run_getnav --------------------------------------------

## (6) get heading correction
def run_getheadcorr(opts):
   print(flankline('step 6: get heading correction (if applicable)'))
   workdir =  os.path.join('cal', 'rotate')
   opts['time_angle_file'] = '%s.ang' % opts['dbname']

   if (opts['head_func'] == 'ashrot'):

      pqlist =  def_qlist('run ashrot?', opts['auto'])       # set up query
      pycmd = [ashrot, (opts, )]
      runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])

      workdir =   'nav'
      pqlist =  def_qlist('write ashtech statistics log file?', opts['auto'])
      pycmd = [print_attlog, (opts, )]
      runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])
   else:

      # this function MUST set opts['time_angle_file']
      #pqlist =  def_qlist('run ' + opts['head_func'] + '?', opts['auto'])
      #pycmd = [opts['head_func'], (opts, )]
      pqlist =  def_qlist('extract posmv heading correction?', opts['auto'])
      pycmd = [get_posmvdh, (opts, )]
      runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])



#--- end run_getheadcorr --------------------------------------------

## (7) rotate database for scan timerange
def run_rotate(opts):
   ## first check to see whether we are re-rotating anything
   print(flankline('step 7: rotate'))
   workdir = os.path.join('cal', 'rotate')

   if  opts['steps2rerun'] != '':
      if 'rotate' in steps2rerun: #not the first time: don't test wasrotated
         # notify and rotate by amp and phase if amp~=1 or phase ~= 0
         opts['no_head_corr'] = 1 # do not use heading correction twice
         wasrotated = 0           # make it ask about rotating
         loglist = ['\n', time.asctime()]
         ss = '\n\n\n' +\
              '***********************************************************\n' +\
              '****** about to apply new rotation to current database ****\n' +\
              '************************************************************\n'
         print ss; loglist.append(ss)
         logappend(opts['logfile'], loglist)
   else: #actually test whether it was rotated
      pycmd = [rotated, (opts, )]
      runcmd(pq=yeslist(), pycmd=pycmd, workdir=workdir,
                                            logfile=opts['logfile'])
      wasrotated = opts['returnval']


   if wasrotated:
      print  '\n\n\n' +\
           '************************************************************\n' +\
           '********* some of these data are already rotated************\n' +\
           '************************************************************\n'

   # now, regardless, run rotate on loaded time range
   workdir = os.path.join('cal', 'rotate')
   cntfile = 'rotate.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [rotate, (opts, cntfile)]
   sqlist =  def_qlist('run rotate?', opts['auto'])
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'rotate'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd,
          workdir=workdir, logfile=opts['logfile'])

# --- end run_rotate --------------------------------------------

## (8a) run xducerxy
def run_xducerxy(opts):
   print(flankline('step 8a: run xducerxy (gga and xducer positions differ)'))
   #now run xducerxy
   workdir = 'nav'
   pqlist =  def_qlist('run xducerxy?', opts['auto'])
   pycmd = [xducerxy, (opts, )]
   runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])

   if os.path.exists(os.path.join(workdir, opts['dbname'] + '.agt')):
      opts['fixfile'] = opts['dbname'] + '.agt'
      print '  NEW fixfile:  %s\n' %opts['fixfile']
   else:
      qa = yn_query('continue??', 'yq',
                    default=qa_default, auto=opts['auto'])

#--- end run_xducerxy --------------------------------------------

## (8) run adcpsect
def run_adcpsect(opts):
   print(flankline('step 8: run adcpsect (for calib)'))
   #now run adcpsect (as_nav)
   workdir = 'nav'
   cntfile = 'as_nav.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [as_nav, (opts, cntfile)]
   sqlist =  def_qlist('run adcpsect?', opts['auto'])
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'adcpsect'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])

#--- end run_adcpsect --------------------------------------------

## (9) run refabs
def run_refabs(opts):
   # now run refabs
   print(flankline('step 9: run refabs (for calib)'))
   workdir = 'nav'
   cntfile = 'refabs.tmp'
   pycmd = [refabs, (opts, cntfile)]
   sqlist =  def_qlist('run refabs?', opts['auto'])
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'refabs'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])

# --- end run_refabs --------------------------------------------

## (10) run smoothr or refsm
def run_smoothr(opts):
   print(flankline('step 10: get smoothed navigation (for calib)'))
   workdir = 'nav'
   cntfile = 'smoothr.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [smoothr, (opts, cntfile)]
   sqlist =  def_qlist('run smoothr?', opts['auto'])
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'smoothr'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])

def run_refsm(opts):
   workdir = 'nav'
   pqlist =  def_qlist('run refsm for smoothed navigation?', opts['auto'])
   pycmd = [refsm, (opts, )]
   runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])


#--- end run_smoothr and run_refsm ---------------------------------


## (11) putnav
def run_putnav(opts):
   print(flankline('step 11: put smoothed navigation back into database'))
   workdir = 'nav'
   cntfile = 'putnav.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [putnav, (opts, cntfile)]
   sqlist =  def_qlist('run putnav?', opts['auto'])
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'putnav'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])

#--- end putnav --------------------------------------------

## (12) reference layer plots
def run_refplots(opts):
   print(flankline('step 12: plot reference layer'))
   # plot reference layer
   workdir = 'nav'
   pqlist =  def_qlist('make reference layer plots?', opts['auto'])
   pycmd = [refplot, (opts, )]
   runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])

#--- end run_refplots ----------------------------------------------------

## (13) -- find pflags automatically
def run_findpflags(opts):
   print(flankline('step 13: automatically find profile flags'))
   print 'find profile flags'
   print(flankline('***'))

   workdir = 'edit'
   pqlist =  def_qlist('automatic editing: find profile flags?', opts['auto'])
   opts['returnval'] = 0 #set this as 0; set to 1 if aedit is run
   pycmd = [aedit, (opts, )]
   runcmd(pq=pqlist, pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])

#--- end run_findpflags--------------------------------------------

## (14) apply editing (all kinds)
def run_applyedit(opts):
   print(flankline('step 14: apply all profile flags to database'))
   if (opts['auto'] == 1):
      qa_default = 'y'
   else:
      qa_default = 'n'
   qa = yn_query('apply all profile flags to database?', 'ynq',
                 default=qa_default, auto=opts['auto'])

   if qa:
      workdir = 'edit'
      startdir = os.getcwd()
      print 'starting in:\n%s' %startdir
      os.chdir(workdir)
      ## start the list of strings which will be appended to the logfile
      loglist = ['\n', time.asctime()]
      logfile = opts['logfile']

      ss =  'executing commands from:\n%s' %workdir
      print ss; loglist.append(ss)

      ## go get the files first
      dbupdate_files = glob.glob('*bottom*.asc')
      dbupdate_files = dbupdate_files + glob.glob('*badprf*.asc')
      dbupdate_files = dbupdate_files + glob.glob('*badtim*.asc')
      badbinfiles = glob.glob('*badbin*.asc')


      cntfile = 'setflags.tmp'
      dbpath   = os.path.join('..', 'adcpdb', opts['dbname'])
      dbupdate = os.path.join(opts['bindir'], 'dbupdate')
      badbin   = os.path.join(opts['bindir'], 'badbin')
      set_lgb  = os.path.join(opts['bindir'], 'set_lgb')
      setflags = os.path.join(opts['bindir'], 'setflags')

      # run dbupdate for bottom and badprofile files
      for ii in range(0, len(dbupdate_files)):
         syscmd = '%s %s %s' % (dbupdate, dbpath, dbupdate_files[ii])
         ss =  'about to run %s' %syscmd
         print ss; loglist.append(ss)
         os.system(syscmd)

      # run badbin
      for ii in range(0, len(badbinfiles)):
         syscmd = '%s %s %s' % (badbin, dbpath, badbinfiles[ii])
         ss =  'about to run %s' %syscmd
         print ss; loglist.append(ss)
         os.system(syscmd)

      # set_lgb
      syscmd = '%s %s' % (set_lgb, dbpath)
      ss =  'about to run %s' %syscmd
      print ss; loglist.append(ss)
      os.system(syscmd)

      # set_flags
      syscmd = '%s %s' % (setflags, cntfile)
      ss =  'about to run %s' %syscmd
      print ss; loglist.append(ss)
      os.system(syscmd)

      ss =  'changing directories back to %s' %startdir
      print ss; loglist.append(ss)


      # write to log file in edit/ subdirectory
      logappend('edit.log', loglist)

      #now chdir back to original directory
      os.chdir(startdir)
      logappend(logfile, loglist)


#-- end run_applyedit --------------------------------------------

## (16) list temperature and plot it
def run_lsttemp(opts):
   opts['returnval'] = 0
   print(flankline('step 16: list temperature and plot it'))
   workdir = 'edit'
   cntfile = 'lst_temp.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [lst_temp, (opts, cntfile)]
   sqlist =  def_qlist('run lst_temp?', opts['auto'])
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'lst_temp'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist, syscmd=syscmd,
          workdir=workdir,
          logfile=opts['logfile'])

   if opts['returnval']:
   ## plot temperature
      workdir = 'edit'
      pycmd = [plottemp, (opts, )]
      runcmd(pq=yeslist(), pycmd=pycmd, workdir=workdir,logfile=opts['logfile'])


#--- end run_lsttemp --------------------------------------------


## (17) run water track and bottom track calibrations
def run_calib(opts):
   print(flankline('step 17: bottom track and watertrack calibrations'))
   ## (17a: bottom track)
   ## (17a.1)now run lst_btrk
   workdir = os.path.join('cal', 'botmtrk')
   cntfile = 'lst_btrk.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [lst_btrk, (opts, cntfile)]
   sqlist =  def_qlist('run botmtrack and watertrack calibrations?', opts['auto'])
   syscmd = '%s %s ' %  (os.path.join(opts['bindir'], 'lst_btrk'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=sqlist,
          syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])

   ## (17a.2) now run refabsbt
   workdir = os.path.join('cal', 'botmtrk')
   cntfile = 'refabsbt.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [refabsbt, (opts, cntfile)]
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'refabsbt'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(),
          syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])

   ## (17a.3) now do the bottom track calculation
   workdir = os.path.join('cal', 'botmtrk')
   pycmd = [btcaluv, (opts, )]
   runcmd(pq=yeslist(), pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])


   ## (17b: water track)
   ## (17b.1) run adcpsect (as_nav)
   workdir = 'nav'
   cntfile = 'as_nav.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [as_nav, (opts, cntfile)]
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'adcpsect'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(),
          syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])

   ## (17b.2) now run watertrack timeslip
   workdir = os.path.join('cal', 'watertrk')
   cntfile = 'timslip.tmp'
   file_msg(os.path.join(workdir, cntfile))
   pycmd = [timslip, (opts, cntfile)]
   syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'timslip'), cntfile)
   runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(),
          syscmd=syscmd, workdir=workdir,
          logfile=opts['logfile'])

   ## (17b.c) do the water track calculation
   workdir = os.path.join('cal', 'watertrk')
   pycmd = [adcpcal, (opts, )]
   runcmd(pq=yeslist(), pycmd=pycmd, workdir=workdir, logfile=opts['logfile'])


#--- end run_calib --------------------------------------------

## (18) get matlab files for plotting
def run_matfiles(opts):
   print(flankline('step 18: make matlab files for plotting'))
   as_vars['loaded_time_range'] = opts['loaded_time_range']  #do all times

   if (opts['auto'] == 1):
      qa_default = 'y'
   else:
      qa_default = 'n'

   qa = yn_query('make matlab files for plotting?', 'ynq',
                 default=qa_default, auto=opts['auto'])
   if qa:
      #force all yn_query answers to be yes

      # go to vector/ directory, make timegrid...
      workdir = 'vector'
      cntfile = opts['afilebase'] + '.tmg'
      as_vars['timegrid'] = 60
      pycmd = [timegrid, (as_vars, cntfile)]
      syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'timegrid'), cntfile)
      runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(), syscmd=syscmd,
             workdir=workdir, logfile=opts['logfile'])
      as_vars['gridfilestring'] = as_vars['timegridstring']
      # ... run adcpsect to get the vector matlab file
      cntfile = 'as_vect.tmp'
      pycmd = [as_vect, (as_vars, cntfile)]
      syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'adcpsect'), cntfile)
      runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(), syscmd=syscmd,
             workdir=workdir, logfile=opts['logfile'])
      print 'done making vector matfiles'
      time.sleep(1)

      # go to contour/ directory, make timegrid...
      workdir = 'contour'
      cntfile = opts['afilebase'] + '.tmg'
      as_vars['timegrid'] = 15
      pycmd = [timegrid, (as_vars, cntfile)]
      syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'timegrid'), cntfile)
      runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(), syscmd=syscmd,
             workdir=workdir,  logfile=opts['logfile'])
      as_vars['gridfilestring'] = as_vars['timegridstring']
      # ... run adcpsect to get the vector matlab file
      cntfile = 'as_cont.tmp'
      pycmd = [as_cont, (as_vars, cntfile)]
      syscmd = '%s %s' %  (os.path.join(opts['bindir'], 'adcpsect'), cntfile)
      runcmd(pq=yeslist(), pycmd=pycmd, sq=yeslist(), syscmd=syscmd,
             workdir=workdir,  logfile=opts['logfile'])
      print 'done making contour matfiles'
      time.sleep(1)

#--- end run_matfiles --------------------------------------------


#=============================================================================


#provide a way to run this independently

if __name__ == '__main__':


   ## possible overrides (instead of commandline options)

#   startopts = {'yearbase'    :    2001,
#                'no_head_corr':     1,
#                'use_refsm'   :     1,
#                'dbname'      :    'ademo',
#                'varvals'    :     1,
#                'nozap'       :     0,
#                'datatype'    :     'pingdata',
#                'auto'        :     0,
#                'allow_reload':     1,
#                'progdir'     :     '/home/ulili/programs'};



   ## get options from command line

   if len(sys.argv[1:]) == 0:
      usage()


   opts = parse_opts()  # get the defaults
   cmdline_opts = parse_opts(sys.argv[1:])  # get just the values from argv
   if cmdline_opts.has_key('cntfile'):
      try:
         linelist = list()
         lines = file(cmdline_opts['cntfile']).readlines()
         for line in lines:
            line = line.split('#')[0]
            tokens = line.split()
            for token in tokens:
               linelist.append(token)
         cntopts = parse_opts(linelist)
         opts.update(cntopts)
      except:
         raise
   opts.update(cmdline_opts)

   opts = opt_paths(opts)
   opts = check_opts(opts)
   opts = get_matcfgstr(opts)

   # set up dictionary for steps to run
   ## steps2run MUST be a tuple
   if  opts['steps2rerun'] != '':
      steps2rerun = opts['steps2rerun'].split(':')
   else:
      steps2rerun = None
   rerun_allowed = ('rotate', 'apply_edit', 'navsteps', 'calib', 'matfiles')

   steps2run = [];

   if not steps2rerun: # running for the first time
      steps2run.append('editsetup') # 1. zapsetup, asetup, aflagit_setup
      if opts['update_gbin']:
         steps2run.append('update_gbin')  # 1a. raw data setup
      steps2run.append('scandata')  # 2. scanping (scanraw), get_time_range
      steps2run.append('loaddata')  # 3. loadping (loadraw), get_time_range
      steps2run.append('setflags')  # 4. setflags
      steps2run.append('getnav')    # 5. ubprint (catnav)

      if (opts['no_head_corr'] == 0):
         steps2run.append('get_headcorr')  # 6. ashrot (or other)
      steps2run.append('rotate')    # 7. rotate
      if (opts['xducer_dx'] + opts['xducer_dy'] != 0):
         steps2run.append('xducerxy') #8a transducer and gga positions differ
      steps2run.append('adcpsect')  # 8. adcpsect for reflayer plots
      steps2run.append('refabs')    # 9. refabs for reflayer plots or smoothr
      steps2run.append('smoothnav') #10. smoothr or refsm
      steps2run.append('putnav')    #11. putnav (using smoothr or refsm)
      steps2run.append('refplots')  #12. plot refl
      if opts['find_pflags']:
         steps2run.append('find_pflags')#13. find profile flags (autoedit)
         steps2run.append('apply_edit') #14. apply editing
         steps2run.append('navsteps')   #15. adcpsect, refabs, smoothr,
                                        #  (refsm if appropriate), putnav
                                        #  refplots
      steps2run.append('lst_temp')  #16. run lst_temp and plot temperature
      steps2run.append('calib')     #17. botmtrk (refabsbt, btcaluv)
      steps2run.append('matfiles')  #18. adcpsect for vector and contour
   else:
      steps2run = []      #start with a blank, add steps to rerun
      # assume this is a "rerun" step.  look for options
      for ii in range(0,len(steps2rerun)):
         if steps2rerun[ii]  in rerun_allowed:
            steps2run.append(steps2rerun[ii])
            print ii, ' appending ', steps2rerun[ii]
         else:
            print 'steps2rerun should come from:'
            print 'rotate:apply_edit:navsteps:calib:matfiles'
            print 'option "%s" not matched' % steps2rerun[ii]
            sys.exit()

   print 'about to run these steps (cwd = %s):\n' % (os.getcwd())
   for ii in range(0, len(steps2run)):
      print steps2run[ii]
   print ''


   #regardless, test for dbname
   if opts['dbname'] == None:
      dirblk = glob.glob('adcpdb/*dir.blk')
      # try looking for dir.blk and using that
      if len(dirblk) == 0:
         print 'exiting... MUST SET DATABASE NAME\n'
         sys.exit()
      elif len(dirblk) > 1:
         print 'exiting... MUST SET DATABASE NAME\n'
         sys.exit()
      else:
         opts['dbname'] = os.path.split(dirblk[0])[1][:-7]


   if (opts['dbname'][0] != 'a'):
      print '*****************************************'
      print '**    WARNING: database name does not start with "a""  **  '
      time.sleep(1)


   # regardless, test for yearbase
   if opts['yearbase'] == None:
      print 'exiting... MUST SET YEARBASE\n'
      sys.exit()



   ### more things to define after loading opts and testing for dbname
   opts['afilebase'] = opts['dbname']
   opts['bindir'] =  os.path.join(opts['progdir'],
                                  'codas3', 'bin', opts['binos'])
   opts['wtrk_step'] = 7  #choose between one of: 5,7,9
   opts['btrk_step'] = (opts['wtrk_step'] - 1)/4;
   opts['logfile'] = os.path.join(os.getcwd(), opts['cruisename'] + '.runlog')
   opts['adjust_time'] = 1;  #we do want to correct times when we scan
   opts['min_filter_fraction']=  .5 # make more strict if p-code (eg 0.75)
   opts['filter_hwidth']    = 0.0208 # otherwise use  0.0208  (half hour)
                                # 15 minutes (if p-code GPS)


   ## get the name of the fix file
   if not steps2rerun: #first time through; we have the datatype
      if opts['datatype'] == 'pingdata':
         if opts['fixfile'] == None: #was not specified; use default
            opts['fixfile'] = '%(afilebase)s.ags' % opts # from ubprint
         opts['data_def'] = 'ub_%(ub_type)s.def' % opts
      else:
         if opts['fixfile'] == None: #was not specified; use default
            opts['fixfile'] = '%(afilebase)s.gps' % opts # from catted nav files
         opts['data_def'] = 'vmadcp.def'
      print 'original fix file is '+opts['fixfile']
      opts['orig_fixfile'] = opts['fixfile']
   else: #try for .ags file first, otherwise use .gps
      if os.path.exists(os.path.join('nav', '%(afilebase)s.ags' % opts)):
         navags_exists = 1
      else:
         navags_exists = 0
      if os.path.exists(os.path.join('nav', '%(afilebase)s.gps' % opts)):
         navgps_exists = 1
      else:
         navgps_exists = 0
      if navgps_exists and navags_exists:
         print "ERROR: cannot determine navigation fix file.  must specify\n"
         sys.exit()
      if (not navgps_exists) and (not navags_exists):
         print "ERROR: no navigation fix file ending with '.gps' or '.ags'\n"
         print " must specify fix file\n"
         sys.exit()
      if navags_exists:
         opts['fixfile'] = '%(afilebase)s.ags' % opts # from ubprint
      if navgps_exists:
         opts['fixfile'] = '%(afilebase)s.gps' % opts # from catted navigation


   ## which smoothed navigation file is used with btrk?
   if opts['use_smoothr']:
      opts['smfile'] = '%(afilebase)s.sm' % opts
   elif opts['use_refsm']:
      opts['smfile'] = 'refsm.sm'
   else:
      print 'putnav.py: must choose "use_refsm" or "use_smoothr"\n'
      sys.exit(1)
   print 'smoothed nav file is '+opts['smfile']


   ## get adcpsect parameters
   ## adcpsect output
   as_vars = get_asvars(opts)
   as_vars['afilebase']  = opts['afilebase']
   if opts['varvals']:
      print print_vals(as_vars, '-----  adcpsect variables ------')

   if opts['varvals']:
      print print_vals(opts)
      sys.exit()


   ############## Warnings: ################

   if not steps2rerun:    # if this is a first run
   ## find out whether there are any files

      get_filelist(opts)

      ## diagnostics:
      if (opts['lastfiles'] != 0):
         print 'starting with data files (as seen from scan/ or load/ directory):\n'+\
               string.join(opts['filelist'], '\n') + '\n\n'
         opts['scanfilelist'] = opts['filelist'][ -1*opts['lastfiles'] :]
         print "LASTFILES invoked. using only %d files from end of filelist\n" \
                  % len(opts['scanfilelist'])
      else:
         opts['scanfilelist'] = opts['filelist']

      if (opts['delblks'] != []):
         print 'using delblks = %d'  % opts['delblks']



      if len(opts['scanfilelist']) == 0:
         print 'NO files found'
         sys.exit()
      else:
         print 'using data files (as seen from scan/ or load/ directory):\n'+\
               string.join(opts['scanfilelist'], '\n') + '\n\n'
      # check file suffix against datatype (should catch some problems)
      datatype_mismatch = 0
      for ii in range(0, len(opts['scanfilelist'])):
         fext = os.path.splitext(opts['scanfilelist'][ii])[1][1:]

         if ((opts['datatype'][-3:] in ('lta', 'LTA')) and \
                                               (fext in ('ens', 'ENS'))) or \
            ((opts['datatype'][-3:] in ('ens', 'ENS')) and \
                                               (fext in ('lta', 'LTA'))) :
            datatype_mismatch = 1
            badext = fext
      if datatype_mismatch:
         print 'WARNING: '
         print 'datatype is %s but at least one file has %s extension' % \
               (opts['datatype'], badext)
         sys.exit()



   ###-----------------  entry in log file ...  --------------------###

   ## now that we are starting to process, write the command and the current
   ## variables and values when quick_adcp.py was called:
   print 'logfile is %s' %opts['logfile']

   loglist = ['\n\n\n\n', time.asctime()]
   loglist.append('running quick_adcp.py as follows:')
   loglist.append(string.join(sys.argv,' '))
   loglist.append(print_vals(as_vars, '\n\n-----  adcpsect variables ------'))
   loglist.append(print_vals(opts,'\n\n-----  quick_adcp.py variables and current values ------'))
   loglist.append('configured to run the following steps:')
   for ii in range(0, len(steps2run)):
      loglist.append(steps2run[ii])

   logappend(opts['logfile'], loglist)

   ###----------------- ...  start processing --------------------###

   if opts['hidedisplay'] == 1:
      old_display = os.environ.get('DISPLAY')
      os.environ['DISPLAY'] = ''
   else:
      old_display = 0


   dbchanged = 0  #assume database has not changed
   print '------- starting Matlab engine -------\n'
   logappend(opts['logfile'], opts['matpathlist'])

   opts['M'] = MatlabEngine(opts['matpathlist'])  #start the matlab engine once

   ## (0) make config file for raw processing, if using raw data
   if opts['datatype'] == 'uhdas':
      run_rawsetup(opts)

   ## (0a) run update_gbin if uhdas
   if ('update_gbin' in steps2run):
      run_updategbin(opts)

   ## (1) scan new files if requested
   if ('scandata' in steps2run) or opts['allow_reload']:
      run_scandata(opts)

   ## always find timerange of scanned data
   ## if file exists, read it; if not, create it from .scn file
   print 'getting scan time range'
   workdir = 'scan'
   trfile = '%(afilebase)s.tr' % opts
   timefile = '%(afilebase)s.scn' % opts
   pycmd = [get_time_range, (opts, timefile, trfile)]
   ## list of start_end_time, dd0, dd1 available through opts['returnval']
   runcmd(pq=yeslist(), pycmd=pycmd,  workdir=workdir, logfile=opts['logfile'])
   opts['scn_time_range'] = opts['returnval'][0]
   opts['scn_startdd'] = opts['returnval'][1]
   opts['scn_enddd'] = opts['returnval'][2]

   ## always find timerange of original database
   ## - gets checked in rotate
   ## - might get changed in loaddata
   print 'in quick_adcp.py: after scan, about to look for database times'
   print 'in %s ' % os.getcwd()
   globstr = os.path.join('adcpdb', '%s*.blk' % opts['dbname'])
   print 'looking for databse with %s' % globstr
   blklst = glob.glob(globstr)
   if len(blklst) > 1:
      print 'database found: listing blocks'
      run_listblocks(opts)
      get_lstblk_tr(opts)
      opts['origdb_time_range'] = opts['returnval'][0]
      opts['origdb_startdd'] = opts['returnval'][1]
      opts['origdb_enddd'] = opts['returnval'][2]

      opts['remaining_time_range'] = opts['returnval'][0]
      opts['remaining_startdd'] = opts['returnval'][1]
      opts['remaining_enddd'] = opts['returnval'][2]
      
   else:
      print 'no database found'
      opts['origdb_time_range'] = '1970/01/01  01:00:00 to 1970/01/01  01:01:01'
      opts['origdb_startdd'] = -10000
      opts['origdb_enddd'] =   -10001

      opts['reamaining_time_range'] = '1970/01/01  01:00:00 to 1970/01/01  01:01:01'
      opts['remaining_startdd'] = -10000
      opts['remaining_enddd'] =   -10001

   print 'in quick_adcp.py: before load, just looked for database times'
   print 'origdb_time_range is %s ' %(opts['origdb_time_range'])
   print '(%f to %f)' %(opts['origdb_startdd'], opts['origdb_enddd'])


   ## (2) load pingdata
   if 'loaddata' in  steps2run:
      dbchanged = run_loaddata(opts)


   ## ALWAYS get time range of whole database
   db_prefix = opts['dbname']
   if  (not os.path.exists(os.path.join(db_prefix + '.lst'))) or \
           (os.path.exists(os.path.join(db_prefix + '.lst')) and \
      os.path.getmtime(db_prefix+'dir.blk') > os.path.getmtime(db_prefix+'.lst')):
      dbchanged = 1

   ## list blocks if necessary
   if dbchanged:
      run_listblocks(opts)
      get_lstblk_tr(opts)
      opts['loaded_time_range'] = opts['returnval'][0]
      opts['loaded_startdd']    = opts['returnval'][1]
      opts['loaded_enddd']      = opts['returnval'][2]


   print 'loaded_time_range is %s ' %(opts['loaded_time_range'])
   print '(%f to %f)' %(opts['loaded_startdd'], opts['loaded_enddd'])


   ## list configuration changes on database
   run_lstconfig(opts)


   #### (3) get ready for autoedit:
   ## make setup.m
   if 'editsetup' in  steps2run:
      run_editsetup(opts)

   ## (4) run setflags
   if 'setflags' in steps2run:
      run_setflags(opts)


   ## (5) get navigation fixes
   if 'getnav' in  steps2run:
      run_getnav(opts)

   ## (6) now get ashtech heading correction
   if ('get_headcorr' in steps2run): # and (opts['datatype'] == 'pingdata'):
      run_getheadcorr(opts)

   ## (7) rotate
   if 'rotate' in steps2run:
      run_rotate(opts)

   ## (8a) xducerxy
   if 'xducerxy' in steps2run:
      #run with original fix file, change afterwards
      run_xducerxy(opts)

   ## (8) adcpsect
   if 'adcpsect' in steps2run:
      run_adcpsect(opts)

   ## (9) refabs
   if 'refabs' in steps2run:
      run_refabs(opts)

   ## (10) smoothed navigation
   if ('smoothnav' in  steps2run):
      run_smoothr(opts)

   if ('smoothnav' in  steps2run) and opts['use_refsm']:
      run_refsm(opts)

   ## (11) now run putnav
   if 'putnav' in steps2run:
      run_putnav(opts)

   ## (12) plot ref layer (needed smoothr output to run)
   if 'refplots' in steps2run:
      run_refplots(opts)

   ## (13) find profile flags (autoedit)
   if 'find_pflags' in steps2run:
      run_findpflags(opts)


   ## (14) apply editing
   if 'apply_edit' in  steps2run:
      run_applyedit(opts)

   ## (15) rerun nav steps
   if 'navsteps' in steps2run:

      print(flankline('step 15: rerun nav steps, redo reflayer plots'))
      if (opts['xducer_dx'] + opts['xducer_dy'] != 0):
      #run with original fix file, change afterwards
         opts['fixfile'] = opts['orig_fixfile']
         run_xducerxy(opts)
         opts['fixfile'] = '%(afilebase)s.agt' % opts
         print '=====> NOTE: all first-run steps following this will \n'
         print '             be run using fix file %s\n' %opts['fixfile']
      run_adcpsect(opts)
      run_refabs(opts)
      run_smoothr(opts)
      if opts['use_refsm']:
         run_refsm(opts)
      run_putnav(opts)
      run_refplots(opts)


   ## (16) now run lst_temp
   if 'lst_temp' in steps2run:
      run_lsttemp(opts)

   ## (17) now run bottom track and water track calibrations
   if 'calib' in steps2run:
      run_calib(opts)

   ## (18) make matlab files
   if 'matfiles' in steps2run:
      run_matfiles(opts)


   if old_display:
      os.environ['DISPLAY'] = old_display
   else:
      try:
         del os.environ['DISPLAY']
      except:
         pass


