#!/usr/bin/env python
# script to create named_uv.mat named_xy.mat files with either
# the standard gridding of vector or contour type,
#

'''
Script to call adcpsect suitable for vector or contour output.

    typical usage:
    mk_mat.py [-y yearbase]  [-d dbpath] [-r rangestring]  [-z startz] \
              [-o outdir]  [-t mins] [-v] [-c]  [-p] [-h]

      Runs timegrid, then makes vector and contour outputs in the
      current directory using that timegrid, with every bin output
      (for contour) or vertically averaged (vector) from startz to
      75m, and in 50m bins to 425.

      Required fields are yearbase, database name, and time range  (or [-p])

      Switches can override defaults.

           -y or --yearbase : 4-digit yearbase  (REQUIRED)
           -d    --dbpath   : relative path to database name, eg:
                            :   ../adcpdb/a0102        (includes dbname)
           -r    --timerange: time range to extract data, quoted string such as
                            :    "2001/04/02 14:32:21 to 2001/04/10 07:21:27"
           -g    --pgmin    : minimum allowable percent good (default 50)
           -z    --zstart   : shallowest bin to start either dataset.
                            :   (should be greater than or equal to:
                            :   [transducer_depth + blank + (bin+pulse)/2])
                            :   default is 31
           -Z    --zend     : maximum depth of contour output
           -b    --binvect  : vector bin edges (string) QUOTED
                            :    eg. "31 50 75 125 175 225 275 325 375 425"
           -c    --cstep    : contour step size
           -o    --outdir   : relative path to output directory (default ./)
           -t    --timegrid : number of minutes to average (default is 15)
                            :   (it is probably pointless to decrease this,
                            :     but the setting to 0 outputs every profile)
                            :
           -V    --vectonly : vector output only (vect_xy.mat, vect_xy.mat)
           -C    --contonly : contour output only (cont_xy.mat, cont_xy.mat)
           -h    --help     : print this message and exit
           -v    --verbose  : print variables and be verbose


       special instance of "polar" ships (Palmer and Gould): turn on -p
                            :  to use this setup (which assumes lots of
                            :  path information) and guesses yearbase,
                            :  dbname, and timerange from available
                            :  information.
           -D    --directory: root cruise directory (where scan/ adcpdb/ are)
                            :     assumed to be /home/adcp/adcp (a link to the
                            :     current cruiseid) or /home/adcp/science
           eg (polar): mk_mat.py  -p --vectonly -t60 --verbose

'''

#JH 2001/04/13
## significant gotcha: if running under cron, there is nothing in here that
##    will set paths to programs/codas3/bin/lnx (or whatever)

import sys, os, string, re, glob, shutil, getopt, pwd

def get_yearbase(dir, globstr):
   adcpfiles = glob.glob(os.path.join(dir,globstr))
   if (adcpfiles == []):
      print 'no ens files exist in ', dir, 'with glob ', ensglob
      sys.exit(1)
   try:
      yearbase = string.atoi(os.path.split(adcpfiles[0])[1][4:8])
   except:
      print 'cannot convert <', adcpfiles[1][4:8], '> to yearbase number'
      sys.exit()
   return yearbase
#end of get_yearbase ----------

def get_dbpath():
   dirblk = glob.glob(os.path.join(home, directory, 'adcpdb', '*dir.blk'))[0]
   dbname =string.split(dirblk, '/')[-1:][0][0:-7]  #split path, take the
                                                    # last element (turns
                                                    # it into a string)
                                                    # take from 0th char
                                                    # to 7 from the end
   return  os.path.join(home, directory, 'adcpdb', dbname)
# end of get_dbpath ----------

def get_timerange(dirbase, subdir = 'scan'):
   trfiles = glob.glob(os.path.join(dirbase, subdir, '*.tr'))
   trlist = []
   for file in trfiles:
      fin = open(file, 'r')
      tempstr = fin.readline()
      trlist.append(tempstr)
      fin.close();
   trlist.sort
   tr1 = string.split(trlist[0])[0:2]
   tr2 = string.split(trlist[-1:][0])[-2:]
   tr = tr1
   tr.insert(len(tr), 'to')
   tr.insert(len(tr), tr2[0])
   tr.insert(len(tr), tr2[1])
   return string.join(tr)
# end of get_timerange ----------

def usage():
   print __doc__
# end of usage   ----------

#adcpsect for vector
asvect_cnt = '''
  dbname:             %(dbpath)s
  output:             %(outdir)s/%(vbasename)s
  step_size:          1        /* must be 1 for navigation */
  ndepth:             128
  time_ranges:        %(ss_string)s
  year_base=          %(yearbase)d

  option_list:
    pg_min=                %(pgmin)s
    reference:             final_ship_ref
    regrid:                average
                           depth
      grid_list number=    %(vector_ngrid)d
        boundaries:       %(binvect)s
    vector:
      minimum_npts=        2
     flag_mask:        ALL_BITS
       end
     end
%(timegridstring)s
'''

#timegrid
tmgrd_cnt = '''
output: %(outdir)s/adcp.tmg
time_interval: %(timegrid)s
time_range: %(timerange)s
'''


#contour string
ascont_cnt = '''

  dbname:             %(dbpath)s
  output:             %(outdir)s/%(cbasename)s
  step_size:          1
  ndepth:             128
  time_ranges:        %(ss_string)s
  year_base=          %(yearbase)d

  option_list:
    pg_min=          %(pgmin)s
    reference:       final_ship_ref

      regrid:         average
                       depth

                       grid number=    %(contour_ngrid)d
                       origin=         %(zstart)d
                       increment=      %(cstep)d



    flag_mask:             ALL_BITS
      end
  end
%(timegridstring)s

'''



######################### end of function defs #########################


# set some constants
pgmin = 50
outdir = './'
yearbase = -1
vbasename = 'vect'
cbasename = 'cont'
zstart = 31;
zend = 425;
binvect = '%s %s' % (zstart, '50 75 125 175 225 275 325 375 425')
cstep = 10
timegrid = 15
dbpath = ''
timerange = ''
vectonly = 0
contonly = 0
verbose = 0
polar = 0
directory = 'adcp'




# parse options, check whether this is a NBP/LMG case (and get
# dbname from CRUISEID, and yearbase from serially logged ensemble data)


try:
   options, args = getopt.getopt(sys.argv[1:], 'y:d:t:z:Z:b:c:o:t:VCphD:v',
                        ['yearbase=', 'dbpath=', 'timerange=',
                         'zstart=','zend=',  'binvect=', 'cstep=',
                         'outdir=', 'timegrid=',
                         'vectonly', 'contonly', 'polar', 'help',
                         'directory=', 'verbose'])
except getopt.GetoptError:
   usage()


for o, a in options:
   if o in ('-y', '--yearbase'):
      yearbase = int(a)
   elif o in ('-d', '--dbpath'):
      dbpath = a
   elif o in ('-D', '--directory'):
      directory = a
   elif o in ('-r', '--timerange'):
      timerange = a
   elif o in ('-z', '--zstart'):
      zstart = int(a)
   elif o in ('-Z', '--zend'):
      zend = int(a)
   elif o in ('-b', '--binvect'):
      binvect = a
   elif o in ('-c', '--cstep'):
      cstep = int(a)
   elif o in ('-g', '--pgmin'):
      pgmin = int(a)
   elif o in ('-o', '--outdir'):
      outdir = a
   elif o in ('-t', '--timegrid'):
      timegrid = int(a)
   elif o in ('-V', '--vectonly'):
      vectonly = 1
   elif o in ('-C', '--contonly'):
      contonly = 1
   elif o in ('-p', '--polar'):
      polar = 1
   elif o in ('-h', '--help'):
      usage()
   elif o in ('-v', '--verbose'):
      verbose = 1

if (vectonly and contonly):
   print '==========> choose vectonly OR contonly (not both) <=========='
   usage()
   sys.exit(1)



# be smarter
vector_ngrid = len(binvect.split())
contour_ngrid = int(round((zend - zstart)/cstep))+1

if (outdir[-1:] == '/'):           #strip the trailing "/" if it's there
   outdir = outdir[0:-1]

if polar:
   ensdir = '/home/data/adcp'                 # to get yearbase
   ensglob = 'adcp????.???'                   # then use characters 4-8
   home = pwd.getpwnam('adcp')[5]             # was  os.environ['HOME']
   execfile('/home/adcp/plots/setup.py')

   if verbose:
      print 'getting info from polar loop'
   dbpath = get_dbpath()
   if verbose:
      print 'dbpath is ',dbpath
   yearbase = get_yearbase(ensdir, ensglob)
   dirbase = os.path.join(home, directory)
   if (verbose):
      print 'home is ', home
      print 'dirbase is', dirbase
   timerange = get_timerange(dirbase, 'adcpdb')
   outdir = html_dir + '/adcp/data'

else:
   if (dbpath == ''):
      usage()
      print '==========>>> must choose database name <<<=========='
      sys.exit(1)
   if (timerange == ''):
      dirbase =  os.path.join(os.path.split(os.path.split(dbpath)[0])[0])
      if (verbose):
         print 'dirbase is', dirbase
      try:
         timerange = get_timerange(dirbase) #appends 'scan/' automatically

      except:
         usage()
         print  'dbpath is ', dbpath
         print '=====> cannot determine time range from ', scanpath, '<===='
         print '=====> must choose time range manually (-r rangestring) <===='
         sys.exit(1)


   if (yearbase == '') or (yearbase < 0):
      usage()
      print '==========>>> must choose yearbase <<=========='
      sys.exit(1)


if (timegrid == 0):
   ss_string = 'single'
   #must use time range at bottom instead of file, don't run timegrid
   timegridstring = timerange
   run_timegrid = 0
else:
   ss_string = 'separate'
   #use timegrid file at the bottom, must run timegrid
   timegridstring = '%s%s/adcp.tmg' % ('@', outdir)
   run_timegrid = 1


#should have everything by now
if verbose:
   print 'yearbase is ', yearbase
   print 'dbpath is ', dbpath
   print 'timerange is ', timerange
   print 'zstart is ', zstart
   print 'binvect is ', binvect
   print 'cstep is', cstep
   print 'output directory is ', outdir
   print 'cruiseid is in directory ', dirbase
   print 'timegrid is ', timegrid
   print 'vectonly is ', vectonly
   print 'contonly is ', contonly
   print 'polar is ', polar



if (run_timegrid):
   cntfile = os.path.join(outdir, 'timegrid.tmp')
   fid = open( cntfile, 'w')

   fid.write(tmgrd_cnt % vars())
   fid.close()


   try:
      os.system('timegrid ' + cntfile + '> /dev/null ')
      print 'making timegrid'
   except:
      print 'could not run timegrid '


if (not contonly):   # do vector
   cntfile = os.path.join(outdir,'as_vect.tmp')
   fid = open( cntfile, 'w')
   fid.write(asvect_cnt % vars())
   fid.close()
   try:
       os.system('adcpsect ' + cntfile  + '> /dev/null ')
       print 'making vector file'
   except:
      print 'could not run adcpsect for vector'


if (not vectonly):  # do contour
   cntfile = os.path.join(outdir,'as_cont.tmp')
   fid = open( cntfile, 'w')
   fid.write(ascont_cnt % vars())
   fid.close()
   try:
      os.system('adcpsect ' + cntfile  + '> /dev/null ')
      print 'making contour file'
   except:
      print 'could not run adcpsect for vector'

