#!/usr/bin/env python

'''
   Python script for setting up and ADCP processing directory

   USAGE:
      python adcptree.py  processing_directory [codas_dir_root]  [options]
                                   (1)              (2)              (3)
                                    

       (1) required

           processing directory is the name of the processing directory.
           It can be a full path or a relative path.  Good choices are the
           cruise name (for averaged data) or the instrument name (for raw
           data)

       (2) deduced from path to adcptree.py:
           directory which contains codas3 directory of programs
           (should not be necessary to override)

       (3) options are:
       
          -i, --instclass     : instrument type, describes instrument 
                              : choose from: os, bb, nb, son.
                              : Default is "nb"

          -n, --instname      : instrument name, subdirectory in which
                              : raw and gbin files exist.  required
                              : for datatype uhdas; irrelevant otherwise
                              

          -d, --datatype      : choose what kind of data logging (and
                              : hence which kind of data files are being
                              : processed)
                              :
                              : name          what
                              : ----          ----
                              : "uhdas"        implies known directory 
                              :                    structure for access
                              :                    to raw data
                              : "lta", "sta"   VmDAS averages
                              : "enx"          implies VmDAS single-ping
                              :                    data,time-corrected,
                              :                    navigated with attitude
                              : "pingdata"     implies DAS2.48 (or 2.49) NB
                              :                    data, already averaged
                              : "ping"         (same)

          -p, --pingtype      : required for instrument os if datatype
                              :     is uhdas: choose "nb" or "bb"
                              : For all others, only one pingtype is
                              : possible (pingtype defaults to
                              : instrument type)

                         ---------------------------------------------
                         
             --configroot     :   uhdas only (use --verbose for more info)
                              :         default =  "cruise"
             --configpath     :   uhdas only  (use --verbose for more info)
                              :         default = "../raw/config"
           -t, --usetemplates :   default 0 (try to find config files)
                              :       If invoked, does not try to find
                              :       *.m files to copy but copies
                              :       templates from demo

          -h, --help          : print this help
          -v, --verbose       : print notes about uhdas logging
                              :            (see previous options)
          -V, --varvals       : print list of options and their values

USAGE:

UNIX:
 example: (for pingdata:)
    adcptree.py vg0101 

 example (for LTA data:)
    adcptree.py vg0301  --instclass os  --datatype lta
    adcptree.py vg0301  -i  os -d lta

 example (for uhdas OS data:)
    adcptree.py os38b  -i os -n os38 -p bb -d uhdas

DOS: (adcptree_py.bat is a shell to pass the options to adcptree.py)

 example: (for pingdata:)
    adcptree_py vg0101 

 example (for LTA data:)
    adcptree_py vg0301  --instclass os  --datatype lta
    adcptree_py vg0301  -i  os -d lta

'''

## 2004/01 JH: adcptree.py is in transition.
## It should still work for pingdata and lta files (ens never worked well)
## the new uhdas portion has not been tested on a PC

import os, sys, string, glob, shutil, re, filecmp, getopt


def copyall(globstr, fromlist, tolist, forcecopy = 0):
   filelist = glob.glob(os.path.join(fromlist[0], fromlist[1], globstr));
   for filename in filelist:
      newfilebase = os.path.split(filename)[-1]
      if len(tolist) == 2:
         todir = os.path.join(tolist[0], tolist[1])
      else:
         todir = tolist[0]
      fullnewfile = os.path.normpath(os.path.join(todir, newfilebase))
         
      if os.path.exists(fullnewfile):
         print 'file exists: not copying %s to %s' % (newfilebase, todir)
         sys.exit()
      else:
         shutil.copy(filename, fullnewfile)


#---------------------------------

def check_opts(val, name, allowed):
   if val not in allowed:
      print ' '
      print __doc__
      print '\n\n\nname <%s> is unacceptable.\n' % (val)
      print 'choose one of the following:\n'
      print allowed
      sys.exit()

#---------------------------------

def varvals(opts):
   strlist = []
   keys = opts.keys()           ## need to do this in two lines:
   keys.sort()                  ## (1) get keys, (2) sort it
   
   for key in keys:
      s = '%s   %s' % (string.ljust(key, 30), str(opts[key]))
      strlist.append(s)
   s = '\n\n'
   strlist.append(s)
   
   print ' '
   print(string.join(strlist, '\n'))

#----------------------------------

def usage():
   print __doc__
   sys.exit()

#---------------------------------

def shortusage():
   ustr = '''
   Python script for setting up and ADCP processing directory

   USAGE:
      python adcptree.py  processing_directory [codas_dir_root]  [options]
                                   (1)              (2)              (3)
                                    
       (1) required (name of the processing directory, eg vg0401)
       (2)    directory which contains codas3 directory of programs
              default: deduced from location of adcptreey.py, PROGRAMS in:
                       PROGRAMS/codas3/bin/scripts/adcptree.py
                       
       (3) switches are:
          -d, --datatype      : choose what kind of data logging
                              :     (lta, sta, pingdata) pre-averaged data
                              :     (enx, ens, uhdas) single-ping data
          -i, --instclass     : instrument type, describes instrument 
                              :    choose from: os, bb, nb, son.
                        --------------------------------------------
          -h, --help          : print more help about defaults and usage
                              :   (especially for enx, ens, uhdas)
          -s, --varvals      : print list of options and their values

USAGE:

 example: (for pingdata:)
    adcptree.py vg0101 

    or on a PC:
    python adcptree.py vg0101 

 example (for LTA data:)
    adcptree.py vg0301  --datatype lta

    or on a PC:
    python adcptree.py vg0301   --datatype lta

'''

   print ustr
   

   

#---------------------------------

def print_uhdasnotes():
   note_str = '''
                   ---------  uhdas notes: ----------
      Processing parameters and directory information are stored in 
      files ending with 
                 "_cfg.m" (for path information) and 
                 "_proc.m" (for instrument and processing information)
                 "_disp.m" (for display during the cruise)
     
      If a source of these files is specified, they will be copied to
      the config subdirectory.  If there is no source specified,
      templates will be left in the config subdirectory.  The
      templates can be used to created appropriate files for
      processing.  The default path to search for a source is in the
      "raw/config" directory of the UHDAS directory tree. The default
      prefix is "cruise".

      In addition to these configuration files, a matlab program defining 
      the instrument class and pingtype ("instping.m") is created so
      processing programs know what parts of the configuration files 
      to apply.
'''
   print note_str


#---------------------------------

def populate(opts):
   demopath    = opts['demopath']
   procdir     = opts['procdir']
   docpath     = opts['docpath']
   codasroot   = opts['codasroot']
   datatype    = opts['datatype']
   instclass   = opts['instclass']
   instname    = opts['instname']
   
   
   dirlist = (
      'adcpdb',
      'cal',
      'cal/watertrk',
      'cal/botmtrk',
      'cal/heading',
      'cal/rotate',
      'contour',
      'edit',
      'grid',
      'load',
      'nav',
      'quality',
      'ping',
      'scan',
      'stick',
      'vector')
   
   
   for dirname in dirlist:
      fulldir = os.path.join(procdir, dirname)
      if os.path.exists(fulldir):
         print 'directory exists: cannot make %s' %(fulldir)
         sys.exit()
      else:
         os.makedirs(fulldir, 0777)
         print 'making directory %s' %(fulldir)


   ## copy glob             fromdir frompath          todir         topath
   copyall('*.cnt',        (demopath,'adcpdb'),       (procdir, 'adcpdb'))
   copyall('*.def',        (demopath,'adcpdb'),       (procdir, 'adcpdb'))
   copyall('*.cnt',        (demopath,'cal/watertrk'), (procdir, 'cal/watertrk'))
   copyall('*.m',          (demopath,'cal/watertrk'), (procdir, 'cal/watertrk'))
   copyall('*.cnt',        (demopath,'cal/botmtrk'),  (procdir, 'cal/botmtrk'))
   copyall('*.m',          (demopath,'cal/heading'),  (procdir, 'cal/heading'))
   copyall('*.cnt',        (demopath,'cal/rotate'),   (procdir, 'cal/rotate'))
   copyall('*.m',          (demopath,'cal/rotate'),   (procdir, 'cal/rotate'))
   copyall('adcpsect.cnt', (demopath,'contour'),      (procdir, 'contour'))
   copyall('contour.cpa',  (demopath,'contour'),      (procdir, 'contour'))
   copyall('*.cnt',        (demopath,'edit'),         (procdir, 'edit'))
   copyall('*.m',          (demopath,'edit'),         (procdir, 'edit'))
   copyall('*.m__',        (demopath,'edit'),         (procdir, 'edit'))
   copyall('llgrid.cnt',   (demopath,'grid'),         (procdir, 'grid'))
   copyall('timegrid.cnt', (demopath,'grid'),         (procdir, 'grid'))
   copyall('*.cnt',        (demopath,'load'),         (procdir, 'load'))
   copyall('*.cnt',        (demopath,'nav'),          (procdir, 'nav'))
   copyall('*.m',          (demopath,'nav'),          (procdir, 'nav'))
   copyall('*.m__',        (demopath,'nav'),          (procdir, 'nav'))
   copyall('*.m',          (demopath,'ping'),         (procdir, 'ping'))
   copyall('*.cnt',        (demopath,'quality'),      (procdir, 'quality'))
   copyall('*.m',          (demopath,'quality'),      (procdir, 'quality'))
   copyall('*.cnt',        (demopath,'scan'),         (procdir, 'scan'))
   copyall('*.def',        (demopath,'scan'),         (procdir, 'scan'))
   copyall('*.m',          (demopath,'scan'),         (procdir, 'scan'))
   copyall('*.cnt',        (demopath,'stick'),        (procdir, 'stick'))
   copyall('*.m',          (demopath,'stick'),        (procdir, 'stick'))
   copyall('*.cnt',        (demopath,'vector'),       (procdir, 'vector'))

   print '\n\n** data type is %s' % (datatype)

   if datatype in ('lta', 'sta'):
      print 'copying additional files for data type %s\n' % (datatype)
      copyall('load_lta.m',              (otherdemo, 'load'),    (procdir, 'load'))
      copyall('load_lta_manual.m',       (otherdemo, 'load'),    (procdir, 'load'))
#     copyall('readme_lta',              (docpath, 'vmdas_doc'), (procdir,''))
#     copyall('readme.quick_adcp_lta',   (docpath, 'vmdas_doc'), (procdir,''))
#     copyall('readme.vmdas_processing', (docpath, 'vmdas_doc'), (procdir,''))
      copyall('aREADME*.txt',            (docpath, 'edit_doc'),  (procdir, 'edit'))
      copyall('ldcodas.cnt',             (otherdemo, 'load'),    (procdir, 'load'))
      copyall('vmadcp.def',              (otherdemo, 'load'),    (procdir, 'load'))
#     copyall('README_FIRST',            (docpath, 'vmdas_doc'), (procdir,''))

   ## this is in transition and reflects the old way.  
   if (datatype == 'enx'):
      print 'copying additional files for data type %s\n' % (datatype)
      copyall('load_ens.m',              (otherdemo, 'load'),    (procdir, 'load'))
#     copyall('readme_ens',              (docpath, 'vmdas_doc'), (procdir,''))
      copyall('aREADME*.txt',            (docpath, 'edit_doc'),  (procdir, 'edit'))
      copyall('ldcodas.cnt',             (otherdemo, 'load'),    (procdir, 'load'))
      copyall('vmadcp.def',              (otherdemo, 'load'),    (procdir, 'load'))
#     copyall('readme.rawdata',          (docpath, 'vmdas_doc'), (procdir,''))
#     copyall('README_FIRST',            (docpath, 'vmdas_doc'), (procdir,''))


   if (datatype == 'uhdas'):
      print 'copying additional files for data type %s\n' % (datatype)
      copyall('*.m',                     (otherdemo, 'load'),    (procdir,'load'))
      copyall('*__',                     (otherdemo, 'load'),    (procdir,'load'))
      copyall('*.def',                   (otherdemo, 'load'),    (procdir,'load'))

      proc_config_dir = os.path.join(procdir,'config')
      os.mkdir(proc_config_dir)
      print 'config files for raw processing are in %s' % (proc_config_dir)
      shutil.copy(config_cfg, proc_config_dir)
      shutil.copy(config_proc, proc_config_dir)
      shutil.copy(config_disp,proc_config_dir)
      IPD = open(os.path.join(proc_config_dir, 'instping.m'),'w')

      varvals(opts)

      
      ipd_str = '''
      %% processing directory is %(procdir)s
      %% datatype is %(datatype)s
      
      instclass = %(instclass)s;
      instname = '%(instname)s';
      pingtype = '%(pingtype)s';
''' % opts
      IPD.write(ipd_str)
      IPD.close()
   
   #make a link to documentation
   index = open(os.path.join(docpath, 'uhdas_doc','index.html__'),'r')
   index_list = index.readlines()
   index.close()
   for ii in range(0, len(index_list)):
      index_list[ii] = index_list[ii].replace('__progdir__index__',
                                              os.path.join(codasroot, 'index.html'))
   indexout = open(os.path.join(procdir, 'adcp_processing.html'),'w')
   for ii in range(0, len(index_list)):
      indexout.write(index_list[ii])
   indexout.close()


   if (datatype == 'ping'):
      print '** demo %s data are in %s\n' % (datatype, os.path.join(demopath,'ping'))
      
   print 'creating local web page for documentation:'
   print '\nopen either of these files in your web browser'
   print os.path.abspath(os.path.join(procdir, 'adcp_processing.html'))
   print os.path.join(codasroot, 'index.html')
   print '\ndone.'


#####################################################################################


if (len(sys.argv) == 1):
   shortusage()
   sys.exit()
   
opts = {}
   
## parse arguments before options   
if len(sys.argv) == 1:
   usage()

# get the processing directory
if sys.argv[1][0] == '-':
   usage()
   print 'must set processing directory name before using switches'
   sys.exit()
else:
   # 
   opts['procdir'] =  sys.argv[1]
   startswitches = 2

### set the default codasroot; override later if necessary
fullname = sys.argv[0]
if not os.path.abspath(fullname):
   print 'must set option "codasroot" explictly'
   print 'this is not supposed to happen.'
   print 'Please let me know what platform this is so I can fix it.'
   print '     Thanks, Jules    (hummon@hawaii.edu)'
   sys.exit()
   
## Assuming $prog/codas3/bin/script/adcptree.py
[scriptpath, quickname] = os.path.split(fullname)
[binpath, scriptname] = os.path.split(scriptpath)
[codaspath, binname] =  os.path.split(binpath)
[codasroot, codasname] =  os.path.split(codaspath)
opts['codasroot'] = codasroot

### override if neccessary, set startswtiches
if len(sys.argv) > 2:
   if sys.argv[2][0] != '-':
      opts['codasroot'] = sys.argv[2]
      startswitches = 3
   else:
      startswitches = 2

print 'codasroot = ', codasroot

            

###
if not os.path.exists(opts['codasroot']):
   print 'codas root directory (%s) not found' % (opts['codasroot'])
   sys.exit()
if not os.path.exists(os.path.join(opts['codasroot'], 'codas3')):
   print 'codas3 programs not found in root directory'
   print 'i.e. %s not found' % (os.path.join(opts['codasroot'], 'codas3'))
   sys.exit()
      

opts['instclass'] = 'nb'
opts['instname'] = ''
opts['pingtype']    = 'nb'
opts['datatype'] = 'pingdata'
opts['configroot'] = 'cruise'
opts['configpath'] = '../raw/config'
opts['usetemplates'] = 0
opts['help'] = 0
opts['verbose'] = 0
opts['varvals'] = 0


try:
   options, args =  getopt.getopt(sys.argv[startswitches:],
                           'i:n:p:d:vVht',  ['instclass=','instname=',
                                             'pingtype=', 'datatype=',
                                             'usetemplates',
                                            'help', 'verbose', 'varvals',
                                             'configroot=', 'configpath='])
except getopt.GetoptError:  # Before Python 2.x this was getopt.error
   usage()
      
for o, a in options:
   if o in ('-h', '--help'):
      usage()
   elif o in ('-v', '--verbose'):
      opts['verbose'] = 1 
   elif o in ('-V', '--varvals'):
      opts['varvals'] = 1 
   elif o in ('-t', '--usetemplates'):
      opts['usetemplates'] = 1
   elif o in ('-n', '--instname'):
      opts['instname'] = a
   elif o in ('-i', '--instclass'):
      opts['instclass'] = a
   elif o in ('-p', '--pingtype'):
      opts['pingtype'] = a
   elif o in ('-d', '--datatype'):
      opts['datatype'] = a
   elif o in ('--configroot'):
      opts['configroot'] = a
   elif o in ('--configpath'):
      opts['configpath'] = a
      

## check for reasonable match between instrument, pingtype and datatype
check_opts(opts['datatype'], 'datatype',
           ('pingdata', 'ping', 'sta', 'lta', 'enr', 'ens', 'enx', 'uhdas'))
check_opts(opts['instclass'], 'instclass',  ('os','nb','son','bb'))

if opts['datatype'] == 'pingdata':
   opts['datatype'] = 'ping'

## look for wrong acquisition type
if opts['datatype'] == 'ping':
   if opts['instclass'] in ('son', 'os'):
      print 'exiting: datatype "%s" is not consistent with instclass "%s"' %\
                (opts['datatype'], opts['instclass'])
      sys.exit()
   elif opts['datatype'] in ('lta','sta', 'enr', 'ens', 'enx'):
      if opts['instclass'] in ('nb', 'son'):
         print 'exiting: datatype "%s" is not consistent with instrument "%s"' %\
               (opts['datatype'], opts['instclass'])
         sys.exit()

## set up instrument and pingtype matches
if opts['instclass'] == 'os':
   if opts['datatype'] == 'uhdas':
      print 'pingtype is '+opts['pingtype']
      if opts['pingtype'] not in ('nb', 'bb'):
         print 'instrument specified as "os".  pingtype must be "nb" or "bb"'
         sys.exit()
else:
   opts['pingtype'] = opts['instclass']


if opts['datatype'] == 'uhdas':
   if opts['instname'] == '':
      print 'uhdas processing: must name instrument with "--instname"'
      sys.exit()


opts['demopath'] = os.path.join(opts['codasroot'], 'codas3/adcp/demo')
opts['docpath']  = os.path.join(opts['codasroot'], 'codas3/adcp/doc')

### end verbose

if not os.path.isdir(opts['demopath']):
   print 'adcptree.py: cannot access demo source directory %s' % (demopath)
   sys.exit()

if opts['datatype'] in ('lta', 'sta'):
   vmdemopath = os.path.join(opts['codasroot'], 'codas3/adcp/vmdas_demos')
   # set default
   otherdemo = os.path.join(vmdemopath, 'os_ltademo')
   if not os.path.isdir(otherdemo):
      print  'adcptree.py: cannot access source directory %s' % (otherdemo)
      sys.exit()

if opts['datatype'] == 'enx':
   vmdemopath = os.path.join(opts['codasroot'], 'codas3/adcp/vmdas_demos')
   # set default
   otherdemo = os.path.join(vmdemopath, 'os_ensdemo')
   if not os.path.isdir(otherdemo):
      print  'adcptree.py: cannot access source directory %s' % (otherdemo)
      sys.exit()

# add some variables if we are not processing pingdata files
if opts['datatype'] in ('uhdas', 'enr', 'ens'):  #treat these all like uhdas
   otherdemo = os.path.join(opts['codasroot'],'codas3/adcp/uhdas_demo')
   print 'otherdemo is %s' % (otherdemo)
   if not os.path.isdir(otherdemo):
      print  'adcptree.py: cannot access source directory %s' % (otherdemo)
      sys.exit()

   ## check for configpath and configroot
   if not os.path.exists(opts['configpath']):
         print 'directory for config files %s does not exist.' % (opts['configpath'])
         opts['usetemplates'] = 1
   if opts['usetemplates'] == 1:
      print 'using templates'
      config_cfg = os.path.join(otherdemo,  'config', 'cruise_cfg.m__');
      config_proc = os.path.join(otherdemo, 'config', 'cruise_proc.m__');
      config_disp = os.path.join(otherdemo, 'config', 'cruise_disp.m__');
   else:
      configbase = os.path.join(opts['configpath'], opts['configroot'])
      config_cfg =  configbase + '_cfg.m'
      config_proc = configbase + '_proc.m'
      config_disp = configbase + '_disp.m'

      
      if not os.path.exists(config_cfg):
         template =  os.path.join(otherdemo, 'config', 'cruise_cfg.m__')
         print 'config file %s not found:' % (config_cfg)
         print 'using template %s' % (template)
         config_cfg = template
         
      if not os.path.exists(config_proc):
         template =  os.path.join(otherdemo, 'config', 'cruise_proc.m__')
         print 'config file %s not found:' % (config_proc)
         print 'using template %s' % (template)
         config_proc = template
         
      if not os.path.exists(config_disp):
         template = os.path.join(otherdemo, 'config', 'cruise_disp.m__')
         print 'config file %s not found:' % (config_disp)
         print 'using template %s' % (template)
         config_disp = template


   if opts['varvals']:
      varvals(opts)
   if opts['verbose']:
      print_uhdasnotes()


populate(opts)


