/*
 * EventCoordinator.java
 *
 * Created on February 27, 2007, 8:01 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package firebottles;
import java.nio.*;
import java.nio.channels.*;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import CommonThreads.*;
import javax.swing.SwingWorker;
import java.util.List;

//import firebottles.FireBottleParser;
/**
 * Coordinates events from the user interface and worker threads.  
 * 
 * All calls to methods of this class must happen within the user-interface
 * thread, with the exception of main(). 
 *
 * This class is the brains behind the FireBottles application.
 *
 * @author Julia J. Robinson
 */
public class EventCoordinator 
        implements GUICommandListener, SWTCommandListener, SWTReceiverEventListener
{
    private RemoteBottleFireGui theGUI;
    private CharactersReceived chars_recvd;
    private ConnectInformation con_info;
    private BottleFireStatus bottle_fire_status;
    private SocketChannel socket_send;
    private boolean non_sequential_firing_is_supported; //depends on water sampler type
    private int next_bottle_to_be_fired;
    private volatile boolean bHaveStatus;
    ReceiverThread the_receiver_thread;
    
    /** Creates a new instance of EventCoordinator */
    public EventCoordinator(String[] args) 
    {
        System.out.println("Constructing the EventCoordinator");
        con_info = new ConnectInformation();
        if (args.length > 0)
        {
            con_info.SetServerAddress(args[0]);
        }
        if (args.length > 1)
        {
            con_info.SetPortInput(Integer.parseInt(args[1]));
        }
        if (args.length > 2)
        {
            con_info.SetPortOutput(Integer.parseInt(args[2]));
        }
        chars_recvd = new CharactersReceived();
        bottle_fire_status = new BottleFireStatus();
        the_receiver_thread = null;
        bHaveStatus = false;
    }
    /** Further initialization after the object is constructed. */
    public void FinalConstruct(RemoteBottleFireGui gui)
    {
        theGUI = gui;
        OnResetTheGUI();
        (new FireBottlesParser(this, chars_recvd)).execute();
    }
    // various error codes
    private final int ERROR_OUTPUT_NOT_CONNECTED = 1;
    private final int ERROR_SEND_THREW_EXCEPTION = 2;
    private final int ERROR_CONNECT_FAILED       = 3;
    
    private String ErrorCodeToString(int errorCode)
    {
        switch (errorCode)
        {
        case ERROR_OUTPUT_NOT_CONNECTED: 
            return "Output socket is not connected.";
        case ERROR_SEND_THREW_EXCEPTION: 
            return "Send threw an exception.";
        case ERROR_CONNECT_FAILED:
            return "Connect failed.";
        }
        return "";
    }

    // Implemenation of the GUICommandListener interface

    /** Called when the user presses the Connect button.*/
    public int OnButtonConnect()
    {
        // get user's input from GUI
        System.out.println("EventCoordinator.OnButtonConnect()");
        con_info.SetPortInput(theGUI.GetInputPort());
        con_info.SetPortOutput(theGUI.GetOutputPort());
        con_info.SetServerAddress(theGUI.GetServerAddress());
        
        // start up a thread that will do the connections
        OnButtonConnectThread thread = new OnButtonConnectThread();
        if (thread != null)
        {
            // disable Connect and Disconnect buttons.
            theGUI.EnableConnectPanelControls(0);
            thread.execute();
        }
        return 0;
    }
    /** Called when user presses Fire button. */
    public int OnButtonFire()
    {
        System.out.println("EventCoordinator.OnButtonFire()");
        if (!con_info.GetOutputIsConnected())
        {
            // don't do anything
            return ERROR_OUTPUT_NOT_CONNECTED;
        }
        if (non_sequential_firing_is_supported)
        {
            next_bottle_to_be_fired = theGUI.GetNextBottle();
            String tag = "FireBottle";
            String command = "<";
            command += tag;
            command += ">";
            command += Integer.toString(next_bottle_to_be_fired);
            command += "</";
            command += tag;
            command += ">";
            Send(command, true);
        }
        else
        {
            String command = "<FireNextBottle/>";
            Send(command, true);
        }
        theGUI.SetNextBottle(next_bottle_to_be_fired);
        return 0;
    }


    // Implementation of the SWTCommandListener interface

    /** 
     * Called after a status message is received and parsed.
     *
     * The FireBottlesParser places contents of the status message into a
     * BottleFireStatus object, which is copied into an EventStatus object
     * and published.  When ParserThread.process gets the EventStatus object, it
     * calls Execute() which calls this function.
     */
    public int OnStatus(BottleFireStatus bfs)
    {
        bHaveStatus = true;
        System.out.println("EventCoordinator.OnStatus()");
        // copy the status object that came with the message.
        bottle_fire_status = new BottleFireStatus(bfs);
        
        String strWaterSamplerType[] = 
        {
            "<unknown>", 
            "GO 1015", 
            "GO 1016", 
            "Hydro Bios", 
            "IOW", 
            "SBE Carousel", 
            "SBE ECO"
        };
        
        int type = bottle_fire_status.GetWaterSamplerType();
        if (type < 0 || type > 6)
            type = 0;
        
        non_sequential_firing_is_supported = (5 == type || 6 == type || 2 == type);
            
        String strDisplay = strWaterSamplerType[type];
        theGUI.SetWaterSamplerType(strDisplay);
        String strReady = "";
        boolean bEnable = bottle_fire_status.GetFiringIsEnabled();
        if (bEnable)
        {
            theGUI.EnableFire(true);
            strReady = "YES";
            if (non_sequential_firing_is_supported)
            {
                theGUI.EnableNextBottle(true);
            }
            else
            {
                theGUI.EnableNextBottle(false);
            }
        }
        else
        {
            strReady = "NO";
            theGUI.EnableFire(false);
        }
        theGUI.SetNumberOfBottles(bottle_fire_status.GetNumberOfBottles());
        theGUI.SetNumberFired(bottle_fire_status.GetNumberOfBottlesFired());
        System.out.println("GUI.OnStatus - " + bottle_fire_status.toString());
        theGUI.SetBottleFireSequence(bottle_fire_status.GetFiringSequence());
        if (this.non_sequential_firing_is_supported)
        {
            theGUI.SetNextBottle(ComputeNextBottle());
        }
        return 0;
    }

    // confirmation of bottle fire received
    public int OnBottleFired(int bottleNumber)
    {
        System.out.println("EventCoordinator.OnBottleFired - #" + Integer.toString(bottleNumber));
        SendRequestForStatusInformation(true);
        return 0;
    }

    public int OnNextBottleFired()
    {
        System.out.println("EventCoordinator.OnNextBottleFired()");
        SendRequestForStatusInformation(true);
        return 0;
    }

    // parser errors, exceptions, and whatever
    public int OnError(String str_error)
    {
        System.out.println("EventCoordinator.OnError() - " + str_error);
        return 0;
    }

    public int OnError(int error_number)
    {
        System.out.println("EventCoordinator.OnError() - #" + Integer.toString(error_number));
        return 0;
    }

    /** parser thread exited */
    public int OnParserThreadExit()
    {
        System.out.println("EventCoordinator.OnParserThreadExit()");
        // We have to have a parser thread, so, start one up.
        (new FireBottlesParser(this, chars_recvd)).execute();
        return 0;
    }
   
    /** socket receiver thread exited */
    public int OnReceiverThreadExit(String exit_message)
    {
        System.out.println("EventCoordinator.OnReceiverThreadExit()");
        /*
        theGUI.EnableConnectPanelControls(true);
        theGUI.EnableFire(false);
        theGUI.EnableNextBottle(false);
        con_info.SetInputIsConnected(false);
        try
        {
            socket_send.close();
        }
        catch (java.io.IOException e)
        {
            System.out.println(e.toString());
        }
        con_info.SetOutputIsConnected(false);
        */
        the_receiver_thread = null;
        Disconnect(true);
        return 0;
    }
    
    /** entry point for this application */
    public static void main(final String[] args) 
    {
        //Schedule a job for the event-dispatching thread:
        //creating and showing this application's GUI.
        javax.swing.SwingUtilities.invokeLater(new Runnable() 
        {
            public void run() 
            {
                createAndShowGUI(args);
            }
        });
    }
    
    public static void createAndShowGUI(final String[] args)
    {
        EventCoordinator ec = new EventCoordinator(args);
        RemoteBottleFireGui gui = RemoteBottleFireGui.createAndShowGUI(ec);
        ec.FinalConstruct(gui);
    }

    private void OnResetTheGUI() 
    {
        theGUI.SetServerAddress(con_info.GetServerAddress());
        theGUI.SetInputPort(con_info.GetPortInput());
        theGUI.SetOutputPort(con_info.GetPortOutput());
        next_bottle_to_be_fired = 1;
        theGUI.SetNextBottle(next_bottle_to_be_fired);
        theGUI.SetNumberFired(0);
        theGUI.SetBottleFireSequence("");
        non_sequential_firing_is_supported = true;
        theGUI.EnableFire(false);
        theGUI.EnableNextBottle(false);
        theGUI.EnableConnectPanelControls(1);
    }
    
    private int Send(String stringCommandToSend, boolean bOKtoChangeGUI) 
    {
        if (!con_info.GetOutputIsConnected())
        {
            Disconnect(bOKtoChangeGUI);
            return ERROR_OUTPUT_NOT_CONNECTED;
        }
	try
	{
            ByteBuffer buffer = ByteBuffer.allocate(128);

            for (int j = 0; j < stringCommandToSend.length(); ++j)
            {
            //    buffer.putChar(stringCommandToSend.charAt(j));
                byte b = (byte)stringCommandToSend.charAt(j);
                buffer.put(b);
            }
            byte b = 0;
            buffer.put(b);
            buffer.flip();
            socket_send.write(buffer);     
            buffer.clear();
	}
        catch (IOException ex) 
        {
            ex.printStackTrace();
            Disconnect(bOKtoChangeGUI);
            return ERROR_SEND_THREW_EXCEPTION;
        }
        return 0;
    }
    
    private void SendRequestForStatusInformation(boolean bOKtochangeGUI)
    {
         if (!con_info.GetOutputIsConnected())
        {
            // don't do anything
            return;
        }
        Send("<SendStatus/>", bOKtochangeGUI);       
    }

    public void Disconnect(boolean bOKtochangeGUI) 
    {
        System.out.println("EventCoordinator.Disconnect()");
        if (null != the_receiver_thread)
        {
            the_receiver_thread.ForceDisconnect();
            the_receiver_thread = null;
        }
        if (null != socket_send)
        {
            try
            {
                socket_send.close();
            }
            catch(java.io.IOException e)
            {
                System.out.println(e.toString());
            }
            socket_send = null;
        }
        con_info.SetInputIsConnected(false);
        con_info.SetOutputIsConnected(false);
        if (bOKtochangeGUI)
        {
            theGUI.EnableConnectPanelControls(1);
            theGUI.EnableFire(false);
            theGUI.EnableNextBottle(false);
        }
    }
    
    public int OnButtonDisconnect()
    {
        Disconnect(true);
        return 0;
    }
   /**
    * Worker thread, to be launched when Connect button is pressed.
    *
    * Establish a socket connection for sending commands.
    *
    * Launches the receiver thread and waits a few seconds for the status 
    * message. Close the socket and shut down the ReceiverThread if the status
    * message does not come.
    **/     
    public final class OnButtonConnectThread extends SwingWorker<Integer, Integer>
    {
        volatile boolean m_bResult;
        
        OnButtonConnectThread()
        {
            m_bResult = false;
        }
        protected Integer doInBackground() throws InterruptedException 
        {
            m_bResult = false;
            // Establish a socket connection for sending commands - 
            // for use by the user interface thread.
            SocketAddress socket_out_address = new InetSocketAddress(
                    con_info.GetServerAddress(), con_info.GetPortOutput());
            try 
            {
                socket_send = SocketChannel.open(socket_out_address);
                Boolean bConnectionPending = socket_send.isConnectionPending();
                while(socket_send.isConnectionPending())
                {
                    bConnectionPending = socket_send.isConnectionPending();
                }
                Boolean bConnected = socket_send.isConnected();
                con_info.SetOutputIsConnected(bConnected);
            } 
            catch (IOException ioe) 
            {
                System.out.println(ioe.toString());
                con_info.SetOutputIsConnected(false);
                return 0;
            }

            if (con_info.GetOutputIsConnected())
            {
                // Start up a thread for receiving stuff via a socket
                the_receiver_thread = (new ReceiverThread(
                        chars_recvd, con_info, GetEventCoordinator()));
                if (null != the_receiver_thread)
                {
                   the_receiver_thread.execute();
                }
                // Test to see if everything works
                bHaveStatus = false;
                SendRequestForStatusInformation(false);
            }
            // Wait up to 10 seconds for the status message to be received.
            for (int i = 0; i < 100 && !bHaveStatus; ++i)
            {
                try
                {
                    Thread.currentThread().sleep(100);
                }
                catch (Exception e)
                {
                    System.out.println(e.toString());
                }
            }
            System.out.println("OnButtonConnectThread.doInBackground is exiting.");
            if (!bHaveStatus)
            {
                //disconnect everything and return false
                Disconnect(false);
                m_bResult = false;
                return 0;
            }
            m_bResult = true;
            return 0;
        }// end OnButtonConnectThread.doInBackground()
        
        // This is called in the GUI's thread with lists of events that have been
        // published.  This particular thread doesn't publish anything.
        protected void process(List<Integer> eventList) 
        {
           for ( Integer pevt: eventList) 
           {
               System.out.println("OnButtonConnectThread.process() got a " 
                       + Integer.toString(pevt));
           }
        }// end OnButtonConnectThread.process()
        
        protected void done()
        {
             System.out.println("OnButtonConnectThread.done - " + 
                     Boolean.toString(m_bResult));
             ConnectThreadDone(m_bResult);
        }// end OnButtonConnectThread.done
        
    }// end class OnButtonConnectThread

    private int ConnectThreadDone(boolean bConnected) 
    {
        System.out.println("EventCoordinator.ConnectThreadDone ");
        
        if (bConnected)
        {
            System.out.println("EventCoordinator.ConnectThreadDone(true) ");
            // enable the Disconnect button
            theGUI.EnableConnectPanelControls(2); 
            theGUI.EnableNextBottle(non_sequential_firing_is_supported);
        }
        else
        {
            System.out.println("EventCoordinator.ConnectThreadDone(false) ");
            // enable the Connect button
            theGUI.EnableConnectPanelControls(1); 
            theGUI.EnableNextBottle(false);
        }
        return 0;
    }


    /** 
     * Allows an inner class to pass a reference to the EventCoordinator to 
     * another constructor.
     */
    public EventCoordinator GetEventCoordinator() 
    {
        return this;
    }

   /**
    * Determine the number to plug in as next bottle to fire - modify this 
    * function if you want a different firing sequence.
    */
    private int ComputeNextBottle() 
    {
        if (bottle_fire_status.GetNumberOfBottles() != 0)
        {
            return 1 + (bottle_fire_status.GetNumberOfBottlesFired() %
                        bottle_fire_status.GetNumberOfBottles());
        }
        return 0;
    }

}
